// Copyright 2009 The Closure Library Authors. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS-IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

goog.provide('goog.math.LongTest');
goog.setTestOnly('goog.math.LongTest');

goog.require('goog.math.Long');
goog.require('goog.testing.jsunit');

// Interprets the given numbers as the bits of a 32-bit int.  In particular,
// this takes care of the 32-bit being interpretted as the sign.
function toInt32s(arr) {
  for (var i = 0; i < arr.length; ++i) {
    arr[i] = arr[i] & 0xFFFFFFFF;
  }
}

// Note that these are in numerical order.
var TEST_BITS = [
  0x80000000, 0x00000000, 0xb776d5f5, 0x5634e2db, 0xffefffff, 0xffffffff,
  0xfff00000, 0x00000000, 0xfffeffff, 0xffffffff, 0xffff0000, 0x00000000,
  0xfffffffe, 0xffffffff, 0xffffffff, 0x00000000, 0xffffffff, 0xfeffffff,
  0xffffffff, 0xff000000, 0xffffffff, 0xfffeffff, 0xffffffff, 0xffff0000,
  0xffffffff, 0xffff7fff, 0xffffffff, 0xffff8000, 0xffffffff, 0xfffffffe,
  0xffffffff, 0xffffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000001,
  0x00000000, 0x00000002, 0x00000000, 0x00007fff, 0x00000000, 0x00008000,
  0x00000000, 0x0000ffff, 0x00000000, 0x00010000, 0x00000000, 0x00ffffff,
  0x00000000, 0x01000000, 0x00000000, 0x5634e2db, 0x00000000, 0xb776d5f5,
  0x00000000, 0xffffffff, 0x00000001, 0x00000000, 0x0000ffff, 0xffffffff,
  0x00010000, 0x00000000, 0x000fffff, 0xffffffff, 0x00100000, 0x00000000,
  0x5634e2db, 0xb776d5f5, 0x7fffffff, 0xffffffff
];
toInt32s(TEST_BITS);

var TEST_ADD_BITS = [
  0x3776d5f5, 0x5634e2db, 0x7fefffff, 0xffffffff, 0xb766d5f5, 0x5634e2da,
  0x7ff00000, 0x00000000, 0xb766d5f5, 0x5634e2db, 0xffdfffff, 0xffffffff,
  0x7ffeffff, 0xffffffff, 0xb775d5f5, 0x5634e2da, 0xffeeffff, 0xfffffffe,
  0xffeeffff, 0xffffffff, 0x7fff0000, 0x00000000, 0xb775d5f5, 0x5634e2db,
  0xffeeffff, 0xffffffff, 0xffef0000, 0x00000000, 0xfffdffff, 0xffffffff,
  0x7ffffffe, 0xffffffff, 0xb776d5f4, 0x5634e2da, 0xffeffffe, 0xfffffffe,
  0xffeffffe, 0xffffffff, 0xfffefffe, 0xfffffffe, 0xfffefffe, 0xffffffff,
  0x7fffffff, 0x00000000, 0xb776d5f4, 0x5634e2db, 0xffeffffe, 0xffffffff,
  0xffefffff, 0x00000000, 0xfffefffe, 0xffffffff, 0xfffeffff, 0x00000000,
  0xfffffffd, 0xffffffff, 0x7fffffff, 0xfeffffff, 0xb776d5f5, 0x5534e2da,
  0xffefffff, 0xfefffffe, 0xffefffff, 0xfeffffff, 0xfffeffff, 0xfefffffe,
  0xfffeffff, 0xfeffffff, 0xfffffffe, 0xfefffffe, 0xfffffffe, 0xfeffffff,
  0x7fffffff, 0xff000000, 0xb776d5f5, 0x5534e2db, 0xffefffff, 0xfeffffff,
  0xffefffff, 0xff000000, 0xfffeffff, 0xfeffffff, 0xfffeffff, 0xff000000,
  0xfffffffe, 0xfeffffff, 0xfffffffe, 0xff000000, 0xffffffff, 0xfdffffff,
  0x7fffffff, 0xfffeffff, 0xb776d5f5, 0x5633e2da, 0xffefffff, 0xfffefffe,
  0xffefffff, 0xfffeffff, 0xfffeffff, 0xfffefffe, 0xfffeffff, 0xfffeffff,
  0xfffffffe, 0xfffefffe, 0xfffffffe, 0xfffeffff, 0xffffffff, 0xfefefffe,
  0xffffffff, 0xfefeffff, 0x7fffffff, 0xffff0000, 0xb776d5f5, 0x5633e2db,
  0xffefffff, 0xfffeffff, 0xffefffff, 0xffff0000, 0xfffeffff, 0xfffeffff,
  0xfffeffff, 0xffff0000, 0xfffffffe, 0xfffeffff, 0xfffffffe, 0xffff0000,
  0xffffffff, 0xfefeffff, 0xffffffff, 0xfeff0000, 0xffffffff, 0xfffdffff,
  0x7fffffff, 0xffff7fff, 0xb776d5f5, 0x563462da, 0xffefffff, 0xffff7ffe,
  0xffefffff, 0xffff7fff, 0xfffeffff, 0xffff7ffe, 0xfffeffff, 0xffff7fff,
  0xfffffffe, 0xffff7ffe, 0xfffffffe, 0xffff7fff, 0xffffffff, 0xfeff7ffe,
  0xffffffff, 0xfeff7fff, 0xffffffff, 0xfffe7ffe, 0xffffffff, 0xfffe7fff,
  0x7fffffff, 0xffff8000, 0xb776d5f5, 0x563462db, 0xffefffff, 0xffff7fff,
  0xffefffff, 0xffff8000, 0xfffeffff, 0xffff7fff, 0xfffeffff, 0xffff8000,
  0xfffffffe, 0xffff7fff, 0xfffffffe, 0xffff8000, 0xffffffff, 0xfeff7fff,
  0xffffffff, 0xfeff8000, 0xffffffff, 0xfffe7fff, 0xffffffff, 0xfffe8000,
  0xffffffff, 0xfffeffff, 0x7fffffff, 0xfffffffe, 0xb776d5f5, 0x5634e2d9,
  0xffefffff, 0xfffffffd, 0xffefffff, 0xfffffffe, 0xfffeffff, 0xfffffffd,
  0xfffeffff, 0xfffffffe, 0xfffffffe, 0xfffffffd, 0xfffffffe, 0xfffffffe,
  0xffffffff, 0xfefffffd, 0xffffffff, 0xfefffffe, 0xffffffff, 0xfffefffd,
  0xffffffff, 0xfffefffe, 0xffffffff, 0xffff7ffd, 0xffffffff, 0xffff7ffe,
  0x7fffffff, 0xffffffff, 0xb776d5f5, 0x5634e2da, 0xffefffff, 0xfffffffe,
  0xffefffff, 0xffffffff, 0xfffeffff, 0xfffffffe, 0xfffeffff, 0xffffffff,
  0xfffffffe, 0xfffffffe, 0xfffffffe, 0xffffffff, 0xffffffff, 0xfefffffe,
  0xffffffff, 0xfeffffff, 0xffffffff, 0xfffefffe, 0xffffffff, 0xfffeffff,
  0xffffffff, 0xffff7ffe, 0xffffffff, 0xffff7fff, 0xffffffff, 0xfffffffd,
  0x80000000, 0x00000000, 0xb776d5f5, 0x5634e2db, 0xffefffff, 0xffffffff,
  0xfff00000, 0x00000000, 0xfffeffff, 0xffffffff, 0xffff0000, 0x00000000,
  0xfffffffe, 0xffffffff, 0xffffffff, 0x00000000, 0xffffffff, 0xfeffffff,
  0xffffffff, 0xff000000, 0xffffffff, 0xfffeffff, 0xffffffff, 0xffff0000,
  0xffffffff, 0xffff7fff, 0xffffffff, 0xffff8000, 0xffffffff, 0xfffffffe,
  0xffffffff, 0xffffffff, 0x80000000, 0x00000001, 0xb776d5f5, 0x5634e2dc,
  0xfff00000, 0x00000000, 0xfff00000, 0x00000001, 0xffff0000, 0x00000000,
  0xffff0000, 0x00000001, 0xffffffff, 0x00000000, 0xffffffff, 0x00000001,
  0xffffffff, 0xff000000, 0xffffffff, 0xff000001, 0xffffffff, 0xffff0000,
  0xffffffff, 0xffff0001, 0xffffffff, 0xffff8000, 0xffffffff, 0xffff8001,
  0xffffffff, 0xffffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000001,
  0x80000000, 0x00000002, 0xb776d5f5, 0x5634e2dd, 0xfff00000, 0x00000001,
  0xfff00000, 0x00000002, 0xffff0000, 0x00000001, 0xffff0000, 0x00000002,
  0xffffffff, 0x00000001, 0xffffffff, 0x00000002, 0xffffffff, 0xff000001,
  0xffffffff, 0xff000002, 0xffffffff, 0xffff0001, 0xffffffff, 0xffff0002,
  0xffffffff, 0xffff8001, 0xffffffff, 0xffff8002, 0x00000000, 0x00000000,
  0x00000000, 0x00000001, 0x00000000, 0x00000002, 0x00000000, 0x00000003,
  0x80000000, 0x00007fff, 0xb776d5f5, 0x563562da, 0xfff00000, 0x00007ffe,
  0xfff00000, 0x00007fff, 0xffff0000, 0x00007ffe, 0xffff0000, 0x00007fff,
  0xffffffff, 0x00007ffe, 0xffffffff, 0x00007fff, 0xffffffff, 0xff007ffe,
  0xffffffff, 0xff007fff, 0xffffffff, 0xffff7ffe, 0xffffffff, 0xffff7fff,
  0xffffffff, 0xfffffffe, 0xffffffff, 0xffffffff, 0x00000000, 0x00007ffd,
  0x00000000, 0x00007ffe, 0x00000000, 0x00007fff, 0x00000000, 0x00008000,
  0x00000000, 0x00008001, 0x80000000, 0x00008000, 0xb776d5f5, 0x563562db,
  0xfff00000, 0x00007fff, 0xfff00000, 0x00008000, 0xffff0000, 0x00007fff,
  0xffff0000, 0x00008000, 0xffffffff, 0x00007fff, 0xffffffff, 0x00008000,
  0xffffffff, 0xff007fff, 0xffffffff, 0xff008000, 0xffffffff, 0xffff7fff,
  0xffffffff, 0xffff8000, 0xffffffff, 0xffffffff, 0x00000000, 0x00000000,
  0x00000000, 0x00007ffe, 0x00000000, 0x00007fff, 0x00000000, 0x00008000,
  0x00000000, 0x00008001, 0x00000000, 0x00008002, 0x00000000, 0x0000ffff,
  0x80000000, 0x0000ffff, 0xb776d5f5, 0x5635e2da, 0xfff00000, 0x0000fffe,
  0xfff00000, 0x0000ffff, 0xffff0000, 0x0000fffe, 0xffff0000, 0x0000ffff,
  0xffffffff, 0x0000fffe, 0xffffffff, 0x0000ffff, 0xffffffff, 0xff00fffe,
  0xffffffff, 0xff00ffff, 0xffffffff, 0xfffffffe, 0xffffffff, 0xffffffff,
  0x00000000, 0x00007ffe, 0x00000000, 0x00007fff, 0x00000000, 0x0000fffd,
  0x00000000, 0x0000fffe, 0x00000000, 0x0000ffff, 0x00000000, 0x00010000,
  0x00000000, 0x00010001, 0x00000000, 0x00017ffe, 0x00000000, 0x00017fff,
  0x80000000, 0x00010000, 0xb776d5f5, 0x5635e2db, 0xfff00000, 0x0000ffff,
  0xfff00000, 0x00010000, 0xffff0000, 0x0000ffff, 0xffff0000, 0x00010000,
  0xffffffff, 0x0000ffff, 0xffffffff, 0x00010000, 0xffffffff, 0xff00ffff,
  0xffffffff, 0xff010000, 0xffffffff, 0xffffffff, 0x00000000, 0x00000000,
  0x00000000, 0x00007fff, 0x00000000, 0x00008000, 0x00000000, 0x0000fffe,
  0x00000000, 0x0000ffff, 0x00000000, 0x00010000, 0x00000000, 0x00010001,
  0x00000000, 0x00010002, 0x00000000, 0x00017fff, 0x00000000, 0x00018000,
  0x00000000, 0x0001ffff, 0x80000000, 0x00ffffff, 0xb776d5f5, 0x5734e2da,
  0xfff00000, 0x00fffffe, 0xfff00000, 0x00ffffff, 0xffff0000, 0x00fffffe,
  0xffff0000, 0x00ffffff, 0xffffffff, 0x00fffffe, 0xffffffff, 0x00ffffff,
  0xffffffff, 0xfffffffe, 0xffffffff, 0xffffffff, 0x00000000, 0x00fefffe,
  0x00000000, 0x00feffff, 0x00000000, 0x00ff7ffe, 0x00000000, 0x00ff7fff,
  0x00000000, 0x00fffffd, 0x00000000, 0x00fffffe, 0x00000000, 0x00ffffff,
  0x00000000, 0x01000000, 0x00000000, 0x01000001, 0x00000000, 0x01007ffe,
  0x00000000, 0x01007fff, 0x00000000, 0x0100fffe, 0x00000000, 0x0100ffff,
  0x80000000, 0x01000000, 0xb776d5f5, 0x5734e2db, 0xfff00000, 0x00ffffff,
  0xfff00000, 0x01000000, 0xffff0000, 0x00ffffff, 0xffff0000, 0x01000000,
  0xffffffff, 0x00ffffff, 0xffffffff, 0x01000000, 0xffffffff, 0xffffffff,
  0x00000000, 0x00000000, 0x00000000, 0x00feffff, 0x00000000, 0x00ff0000,
  0x00000000, 0x00ff7fff, 0x00000000, 0x00ff8000, 0x00000000, 0x00fffffe,
  0x00000000, 0x00ffffff, 0x00000000, 0x01000000, 0x00000000, 0x01000001,
  0x00000000, 0x01000002, 0x00000000, 0x01007fff, 0x00000000, 0x01008000,
  0x00000000, 0x0100ffff, 0x00000000, 0x01010000, 0x00000000, 0x01ffffff,
  0x80000000, 0x5634e2db, 0xb776d5f5, 0xac69c5b6, 0xfff00000, 0x5634e2da,
  0xfff00000, 0x5634e2db, 0xffff0000, 0x5634e2da, 0xffff0000, 0x5634e2db,
  0xffffffff, 0x5634e2da, 0xffffffff, 0x5634e2db, 0x00000000, 0x5534e2da,
  0x00000000, 0x5534e2db, 0x00000000, 0x5633e2da, 0x00000000, 0x5633e2db,
  0x00000000, 0x563462da, 0x00000000, 0x563462db, 0x00000000, 0x5634e2d9,
  0x00000000, 0x5634e2da, 0x00000000, 0x5634e2db, 0x00000000, 0x5634e2dc,
  0x00000000, 0x5634e2dd, 0x00000000, 0x563562da, 0x00000000, 0x563562db,
  0x00000000, 0x5635e2da, 0x00000000, 0x5635e2db, 0x00000000, 0x5734e2da,
  0x00000000, 0x5734e2db, 0x80000000, 0xb776d5f5, 0xb776d5f6, 0x0dabb8d0,
  0xfff00000, 0xb776d5f4, 0xfff00000, 0xb776d5f5, 0xffff0000, 0xb776d5f4,
  0xffff0000, 0xb776d5f5, 0xffffffff, 0xb776d5f4, 0xffffffff, 0xb776d5f5,
  0x00000000, 0xb676d5f4, 0x00000000, 0xb676d5f5, 0x00000000, 0xb775d5f4,
  0x00000000, 0xb775d5f5, 0x00000000, 0xb77655f4, 0x00000000, 0xb77655f5,
  0x00000000, 0xb776d5f3, 0x00000000, 0xb776d5f4, 0x00000000, 0xb776d5f5,
  0x00000000, 0xb776d5f6, 0x00000000, 0xb776d5f7, 0x00000000, 0xb77755f4,
  0x00000000, 0xb77755f5, 0x00000000, 0xb777d5f4, 0x00000000, 0xb777d5f5,
  0x00000000, 0xb876d5f4, 0x00000000, 0xb876d5f5, 0x00000001, 0x0dabb8d0,
  0x80000000, 0xffffffff, 0xb776d5f6, 0x5634e2da, 0xfff00000, 0xfffffffe,
  0xfff00000, 0xffffffff, 0xffff0000, 0xfffffffe, 0xffff0000, 0xffffffff,
  0xffffffff, 0xfffffffe, 0xffffffff, 0xffffffff, 0x00000000, 0xfefffffe,
  0x00000000, 0xfeffffff, 0x00000000, 0xfffefffe, 0x00000000, 0xfffeffff,
  0x00000000, 0xffff7ffe, 0x00000000, 0xffff7fff, 0x00000000, 0xfffffffd,
  0x00000000, 0xfffffffe, 0x00000000, 0xffffffff, 0x00000001, 0x00000000,
  0x00000001, 0x00000001, 0x00000001, 0x00007ffe, 0x00000001, 0x00007fff,
  0x00000001, 0x0000fffe, 0x00000001, 0x0000ffff, 0x00000001, 0x00fffffe,
  0x00000001, 0x00ffffff, 0x00000001, 0x5634e2da, 0x00000001, 0xb776d5f4,
  0x80000001, 0x00000000, 0xb776d5f6, 0x5634e2db, 0xfff00000, 0xffffffff,
  0xfff00001, 0x00000000, 0xffff0000, 0xffffffff, 0xffff0001, 0x00000000,
  0xffffffff, 0xffffffff, 0x00000000, 0x00000000, 0x00000000, 0xfeffffff,
  0x00000000, 0xff000000, 0x00000000, 0xfffeffff, 0x00000000, 0xffff0000,
  0x00000000, 0xffff7fff, 0x00000000, 0xffff8000, 0x00000000, 0xfffffffe,
  0x00000000, 0xffffffff, 0x00000001, 0x00000000, 0x00000001, 0x00000001,
  0x00000001, 0x00000002, 0x00000001, 0x00007fff, 0x00000001, 0x00008000,
  0x00000001, 0x0000ffff, 0x00000001, 0x00010000, 0x00000001, 0x00ffffff,
  0x00000001, 0x01000000, 0x00000001, 0x5634e2db, 0x00000001, 0xb776d5f5,
  0x00000001, 0xffffffff, 0x8000ffff, 0xffffffff, 0xb777d5f5, 0x5634e2da,
  0xfff0ffff, 0xfffffffe, 0xfff0ffff, 0xffffffff, 0xffffffff, 0xfffffffe,
  0xffffffff, 0xffffffff, 0x0000fffe, 0xfffffffe, 0x0000fffe, 0xffffffff,
  0x0000ffff, 0xfefffffe, 0x0000ffff, 0xfeffffff, 0x0000ffff, 0xfffefffe,
  0x0000ffff, 0xfffeffff, 0x0000ffff, 0xffff7ffe, 0x0000ffff, 0xffff7fff,
  0x0000ffff, 0xfffffffd, 0x0000ffff, 0xfffffffe, 0x0000ffff, 0xffffffff,
  0x00010000, 0x00000000, 0x00010000, 0x00000001, 0x00010000, 0x00007ffe,
  0x00010000, 0x00007fff, 0x00010000, 0x0000fffe, 0x00010000, 0x0000ffff,
  0x00010000, 0x00fffffe, 0x00010000, 0x00ffffff, 0x00010000, 0x5634e2da,
  0x00010000, 0xb776d5f4, 0x00010000, 0xfffffffe, 0x00010000, 0xffffffff,
  0x80010000, 0x00000000, 0xb777d5f5, 0x5634e2db, 0xfff0ffff, 0xffffffff,
  0xfff10000, 0x00000000, 0xffffffff, 0xffffffff, 0x00000000, 0x00000000,
  0x0000fffe, 0xffffffff, 0x0000ffff, 0x00000000, 0x0000ffff, 0xfeffffff,
  0x0000ffff, 0xff000000, 0x0000ffff, 0xfffeffff, 0x0000ffff, 0xffff0000,
  0x0000ffff, 0xffff7fff, 0x0000ffff, 0xffff8000, 0x0000ffff, 0xfffffffe,
  0x0000ffff, 0xffffffff, 0x00010000, 0x00000000, 0x00010000, 0x00000001,
  0x00010000, 0x00000002, 0x00010000, 0x00007fff, 0x00010000, 0x00008000,
  0x00010000, 0x0000ffff, 0x00010000, 0x00010000, 0x00010000, 0x00ffffff,
  0x00010000, 0x01000000, 0x00010000, 0x5634e2db, 0x00010000, 0xb776d5f5,
  0x00010000, 0xffffffff, 0x00010001, 0x00000000, 0x0001ffff, 0xffffffff,
  0x800fffff, 0xffffffff, 0xb786d5f5, 0x5634e2da, 0xffffffff, 0xfffffffe,
  0xffffffff, 0xffffffff, 0x000effff, 0xfffffffe, 0x000effff, 0xffffffff,
  0x000ffffe, 0xfffffffe, 0x000ffffe, 0xffffffff, 0x000fffff, 0xfefffffe,
  0x000fffff, 0xfeffffff, 0x000fffff, 0xfffefffe, 0x000fffff, 0xfffeffff,
  0x000fffff, 0xffff7ffe, 0x000fffff, 0xffff7fff, 0x000fffff, 0xfffffffd,
  0x000fffff, 0xfffffffe, 0x000fffff, 0xffffffff, 0x00100000, 0x00000000,
  0x00100000, 0x00000001, 0x00100000, 0x00007ffe, 0x00100000, 0x00007fff,
  0x00100000, 0x0000fffe, 0x00100000, 0x0000ffff, 0x00100000, 0x00fffffe,
  0x00100000, 0x00ffffff, 0x00100000, 0x5634e2da, 0x00100000, 0xb776d5f4,
  0x00100000, 0xfffffffe, 0x00100000, 0xffffffff, 0x0010ffff, 0xfffffffe,
  0x0010ffff, 0xffffffff, 0x80100000, 0x00000000, 0xb786d5f5, 0x5634e2db,
  0xffffffff, 0xffffffff, 0x00000000, 0x00000000, 0x000effff, 0xffffffff,
  0x000f0000, 0x00000000, 0x000ffffe, 0xffffffff, 0x000fffff, 0x00000000,
  0x000fffff, 0xfeffffff, 0x000fffff, 0xff000000, 0x000fffff, 0xfffeffff,
  0x000fffff, 0xffff0000, 0x000fffff, 0xffff7fff, 0x000fffff, 0xffff8000,
  0x000fffff, 0xfffffffe, 0x000fffff, 0xffffffff, 0x00100000, 0x00000000,
  0x00100000, 0x00000001, 0x00100000, 0x00000002, 0x00100000, 0x00007fff,
  0x00100000, 0x00008000, 0x00100000, 0x0000ffff, 0x00100000, 0x00010000,
  0x00100000, 0x00ffffff, 0x00100000, 0x01000000, 0x00100000, 0x5634e2db,
  0x00100000, 0xb776d5f5, 0x00100000, 0xffffffff, 0x00100001, 0x00000000,
  0x0010ffff, 0xffffffff, 0x00110000, 0x00000000, 0x001fffff, 0xffffffff,
  0xd634e2db, 0xb776d5f5, 0x0dabb8d1, 0x0dabb8d0, 0x5624e2db, 0xb776d5f4,
  0x5624e2db, 0xb776d5f5, 0x5633e2db, 0xb776d5f4, 0x5633e2db, 0xb776d5f5,
  0x5634e2da, 0xb776d5f4, 0x5634e2da, 0xb776d5f5, 0x5634e2db, 0xb676d5f4,
  0x5634e2db, 0xb676d5f5, 0x5634e2db, 0xb775d5f4, 0x5634e2db, 0xb775d5f5,
  0x5634e2db, 0xb77655f4, 0x5634e2db, 0xb77655f5, 0x5634e2db, 0xb776d5f3,
  0x5634e2db, 0xb776d5f4, 0x5634e2db, 0xb776d5f5, 0x5634e2db, 0xb776d5f6,
  0x5634e2db, 0xb776d5f7, 0x5634e2db, 0xb77755f4, 0x5634e2db, 0xb77755f5,
  0x5634e2db, 0xb777d5f4, 0x5634e2db, 0xb777d5f5, 0x5634e2db, 0xb876d5f4,
  0x5634e2db, 0xb876d5f5, 0x5634e2dc, 0x0dabb8d0, 0x5634e2dc, 0x6eedabea,
  0x5634e2dc, 0xb776d5f4, 0x5634e2dc, 0xb776d5f5, 0x5635e2db, 0xb776d5f4,
  0x5635e2db, 0xb776d5f5, 0x5644e2db, 0xb776d5f4, 0x5644e2db, 0xb776d5f5,
  0xffffffff, 0xffffffff, 0x3776d5f5, 0x5634e2da, 0x7fefffff, 0xfffffffe,
  0x7fefffff, 0xffffffff, 0x7ffeffff, 0xfffffffe, 0x7ffeffff, 0xffffffff,
  0x7ffffffe, 0xfffffffe, 0x7ffffffe, 0xffffffff, 0x7fffffff, 0xfefffffe,
  0x7fffffff, 0xfeffffff, 0x7fffffff, 0xfffefffe, 0x7fffffff, 0xfffeffff,
  0x7fffffff, 0xffff7ffe, 0x7fffffff, 0xffff7fff, 0x7fffffff, 0xfffffffd,
  0x7fffffff, 0xfffffffe, 0x7fffffff, 0xffffffff, 0x80000000, 0x00000000,
  0x80000000, 0x00000001, 0x80000000, 0x00007ffe, 0x80000000, 0x00007fff,
  0x80000000, 0x0000fffe, 0x80000000, 0x0000ffff, 0x80000000, 0x00fffffe,
  0x80000000, 0x00ffffff, 0x80000000, 0x5634e2da, 0x80000000, 0xb776d5f4,
  0x80000000, 0xfffffffe, 0x80000000, 0xffffffff, 0x8000ffff, 0xfffffffe,
  0x8000ffff, 0xffffffff, 0x800fffff, 0xfffffffe, 0x800fffff, 0xffffffff,
  0xd634e2db, 0xb776d5f4
];
toInt32s(TEST_ADD_BITS);

var TEST_SUB_BITS = [
  0x00000000, 0x00000000, 0xc8892a0a, 0xa9cb1d25, 0x80100000, 0x00000001,
  0x80100000, 0x00000000, 0x80010000, 0x00000001, 0x80010000, 0x00000000,
  0x80000001, 0x00000001, 0x80000001, 0x00000000, 0x80000000, 0x01000001,
  0x80000000, 0x01000000, 0x80000000, 0x00010001, 0x80000000, 0x00010000,
  0x80000000, 0x00008001, 0x80000000, 0x00008000, 0x80000000, 0x00000002,
  0x80000000, 0x00000001, 0x80000000, 0x00000000, 0x7fffffff, 0xffffffff,
  0x7fffffff, 0xfffffffe, 0x7fffffff, 0xffff8001, 0x7fffffff, 0xffff8000,
  0x7fffffff, 0xffff0001, 0x7fffffff, 0xffff0000, 0x7fffffff, 0xff000001,
  0x7fffffff, 0xff000000, 0x7fffffff, 0xa9cb1d25, 0x7fffffff, 0x48892a0b,
  0x7fffffff, 0x00000001, 0x7fffffff, 0x00000000, 0x7fff0000, 0x00000001,
  0x7fff0000, 0x00000000, 0x7ff00000, 0x00000001, 0x7ff00000, 0x00000000,
  0x29cb1d24, 0x48892a0b, 0x00000000, 0x00000001, 0x3776d5f5, 0x5634e2db,
  0x00000000, 0x00000000, 0xb786d5f5, 0x5634e2dc, 0xb786d5f5, 0x5634e2db,
  0xb777d5f5, 0x5634e2dc, 0xb777d5f5, 0x5634e2db, 0xb776d5f6, 0x5634e2dc,
  0xb776d5f6, 0x5634e2db, 0xb776d5f5, 0x5734e2dc, 0xb776d5f5, 0x5734e2db,
  0xb776d5f5, 0x5635e2dc, 0xb776d5f5, 0x5635e2db, 0xb776d5f5, 0x563562dc,
  0xb776d5f5, 0x563562db, 0xb776d5f5, 0x5634e2dd, 0xb776d5f5, 0x5634e2dc,
  0xb776d5f5, 0x5634e2db, 0xb776d5f5, 0x5634e2da, 0xb776d5f5, 0x5634e2d9,
  0xb776d5f5, 0x563462dc, 0xb776d5f5, 0x563462db, 0xb776d5f5, 0x5633e2dc,
  0xb776d5f5, 0x5633e2db, 0xb776d5f5, 0x5534e2dc, 0xb776d5f5, 0x5534e2db,
  0xb776d5f5, 0x00000000, 0xb776d5f4, 0x9ebe0ce6, 0xb776d5f4, 0x5634e2dc,
  0xb776d5f4, 0x5634e2db, 0xb775d5f5, 0x5634e2dc, 0xb775d5f5, 0x5634e2db,
  0xb766d5f5, 0x5634e2dc, 0xb766d5f5, 0x5634e2db, 0x6141f319, 0x9ebe0ce6,
  0x3776d5f5, 0x5634e2dc, 0x7fefffff, 0xffffffff, 0x48792a0a, 0xa9cb1d24,
  0x00000000, 0x00000000, 0xffffffff, 0xffffffff, 0xfff10000, 0x00000000,
  0xfff0ffff, 0xffffffff, 0xfff00001, 0x00000000, 0xfff00000, 0xffffffff,
  0xfff00000, 0x01000000, 0xfff00000, 0x00ffffff, 0xfff00000, 0x00010000,
  0xfff00000, 0x0000ffff, 0xfff00000, 0x00008000, 0xfff00000, 0x00007fff,
  0xfff00000, 0x00000001, 0xfff00000, 0x00000000, 0xffefffff, 0xffffffff,
  0xffefffff, 0xfffffffe, 0xffefffff, 0xfffffffd, 0xffefffff, 0xffff8000,
  0xffefffff, 0xffff7fff, 0xffefffff, 0xffff0000, 0xffefffff, 0xfffeffff,
  0xffefffff, 0xff000000, 0xffefffff, 0xfeffffff, 0xffefffff, 0xa9cb1d24,
  0xffefffff, 0x48892a0a, 0xffefffff, 0x00000000, 0xffeffffe, 0xffffffff,
  0xffef0000, 0x00000000, 0xffeeffff, 0xffffffff, 0xffe00000, 0x00000000,
  0xffdfffff, 0xffffffff, 0xa9bb1d24, 0x48892a0a, 0x7ff00000, 0x00000000,
  0x7ff00000, 0x00000000, 0x48792a0a, 0xa9cb1d25, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0xfff10000, 0x00000001, 0xfff10000, 0x00000000,
  0xfff00001, 0x00000001, 0xfff00001, 0x00000000, 0xfff00000, 0x01000001,
  0xfff00000, 0x01000000, 0xfff00000, 0x00010001, 0xfff00000, 0x00010000,
  0xfff00000, 0x00008001, 0xfff00000, 0x00008000, 0xfff00000, 0x00000002,
  0xfff00000, 0x00000001, 0xfff00000, 0x00000000, 0xffefffff, 0xffffffff,
  0xffefffff, 0xfffffffe, 0xffefffff, 0xffff8001, 0xffefffff, 0xffff8000,
  0xffefffff, 0xffff0001, 0xffefffff, 0xffff0000, 0xffefffff, 0xff000001,
  0xffefffff, 0xff000000, 0xffefffff, 0xa9cb1d25, 0xffefffff, 0x48892a0b,
  0xffefffff, 0x00000001, 0xffefffff, 0x00000000, 0xffef0000, 0x00000001,
  0xffef0000, 0x00000000, 0xffe00000, 0x00000001, 0xffe00000, 0x00000000,
  0xa9bb1d24, 0x48892a0b, 0x7ff00000, 0x00000001, 0x7ffeffff, 0xffffffff,
  0x48882a0a, 0xa9cb1d24, 0x000f0000, 0x00000000, 0x000effff, 0xffffffff,
  0x00000000, 0x00000000, 0xffffffff, 0xffffffff, 0xffff0001, 0x00000000,
  0xffff0000, 0xffffffff, 0xffff0000, 0x01000000, 0xffff0000, 0x00ffffff,
  0xffff0000, 0x00010000, 0xffff0000, 0x0000ffff, 0xffff0000, 0x00008000,
  0xffff0000, 0x00007fff, 0xffff0000, 0x00000001, 0xffff0000, 0x00000000,
  0xfffeffff, 0xffffffff, 0xfffeffff, 0xfffffffe, 0xfffeffff, 0xfffffffd,
  0xfffeffff, 0xffff8000, 0xfffeffff, 0xffff7fff, 0xfffeffff, 0xffff0000,
  0xfffeffff, 0xfffeffff, 0xfffeffff, 0xff000000, 0xfffeffff, 0xfeffffff,
  0xfffeffff, 0xa9cb1d24, 0xfffeffff, 0x48892a0a, 0xfffeffff, 0x00000000,
  0xfffefffe, 0xffffffff, 0xfffe0000, 0x00000000, 0xfffdffff, 0xffffffff,
  0xffef0000, 0x00000000, 0xffeeffff, 0xffffffff, 0xa9ca1d24, 0x48892a0a,
  0x7fff0000, 0x00000000, 0x7fff0000, 0x00000000, 0x48882a0a, 0xa9cb1d25,
  0x000f0000, 0x00000001, 0x000f0000, 0x00000000, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0xffff0001, 0x00000001, 0xffff0001, 0x00000000,
  0xffff0000, 0x01000001, 0xffff0000, 0x01000000, 0xffff0000, 0x00010001,
  0xffff0000, 0x00010000, 0xffff0000, 0x00008001, 0xffff0000, 0x00008000,
  0xffff0000, 0x00000002, 0xffff0000, 0x00000001, 0xffff0000, 0x00000000,
  0xfffeffff, 0xffffffff, 0xfffeffff, 0xfffffffe, 0xfffeffff, 0xffff8001,
  0xfffeffff, 0xffff8000, 0xfffeffff, 0xffff0001, 0xfffeffff, 0xffff0000,
  0xfffeffff, 0xff000001, 0xfffeffff, 0xff000000, 0xfffeffff, 0xa9cb1d25,
  0xfffeffff, 0x48892a0b, 0xfffeffff, 0x00000001, 0xfffeffff, 0x00000000,
  0xfffe0000, 0x00000001, 0xfffe0000, 0x00000000, 0xffef0000, 0x00000001,
  0xffef0000, 0x00000000, 0xa9ca1d24, 0x48892a0b, 0x7fff0000, 0x00000001,
  0x7ffffffe, 0xffffffff, 0x48892a09, 0xa9cb1d24, 0x000fffff, 0x00000000,
  0x000ffffe, 0xffffffff, 0x0000ffff, 0x00000000, 0x0000fffe, 0xffffffff,
  0x00000000, 0x00000000, 0xffffffff, 0xffffffff, 0xffffffff, 0x01000000,
  0xffffffff, 0x00ffffff, 0xffffffff, 0x00010000, 0xffffffff, 0x0000ffff,
  0xffffffff, 0x00008000, 0xffffffff, 0x00007fff, 0xffffffff, 0x00000001,
  0xffffffff, 0x00000000, 0xfffffffe, 0xffffffff, 0xfffffffe, 0xfffffffe,
  0xfffffffe, 0xfffffffd, 0xfffffffe, 0xffff8000, 0xfffffffe, 0xffff7fff,
  0xfffffffe, 0xffff0000, 0xfffffffe, 0xfffeffff, 0xfffffffe, 0xff000000,
  0xfffffffe, 0xfeffffff, 0xfffffffe, 0xa9cb1d24, 0xfffffffe, 0x48892a0a,
  0xfffffffe, 0x00000000, 0xfffffffd, 0xffffffff, 0xfffeffff, 0x00000000,
  0xfffefffe, 0xffffffff, 0xffefffff, 0x00000000, 0xffeffffe, 0xffffffff,
  0xa9cb1d23, 0x48892a0a, 0x7fffffff, 0x00000000, 0x7fffffff, 0x00000000,
  0x48892a09, 0xa9cb1d25, 0x000fffff, 0x00000001, 0x000fffff, 0x00000000,
  0x0000ffff, 0x00000001, 0x0000ffff, 0x00000000, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0xffffffff, 0x01000001, 0xffffffff, 0x01000000,
  0xffffffff, 0x00010001, 0xffffffff, 0x00010000, 0xffffffff, 0x00008001,
  0xffffffff, 0x00008000, 0xffffffff, 0x00000002, 0xffffffff, 0x00000001,
  0xffffffff, 0x00000000, 0xfffffffe, 0xffffffff, 0xfffffffe, 0xfffffffe,
  0xfffffffe, 0xffff8001, 0xfffffffe, 0xffff8000, 0xfffffffe, 0xffff0001,
  0xfffffffe, 0xffff0000, 0xfffffffe, 0xff000001, 0xfffffffe, 0xff000000,
  0xfffffffe, 0xa9cb1d25, 0xfffffffe, 0x48892a0b, 0xfffffffe, 0x00000001,
  0xfffffffe, 0x00000000, 0xfffeffff, 0x00000001, 0xfffeffff, 0x00000000,
  0xffefffff, 0x00000001, 0xffefffff, 0x00000000, 0xa9cb1d23, 0x48892a0b,
  0x7fffffff, 0x00000001, 0x7fffffff, 0xfeffffff, 0x48892a0a, 0xa8cb1d24,
  0x000fffff, 0xff000000, 0x000fffff, 0xfeffffff, 0x0000ffff, 0xff000000,
  0x0000ffff, 0xfeffffff, 0x00000000, 0xff000000, 0x00000000, 0xfeffffff,
  0x00000000, 0x00000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xff010000,
  0xffffffff, 0xff00ffff, 0xffffffff, 0xff008000, 0xffffffff, 0xff007fff,
  0xffffffff, 0xff000001, 0xffffffff, 0xff000000, 0xffffffff, 0xfeffffff,
  0xffffffff, 0xfefffffe, 0xffffffff, 0xfefffffd, 0xffffffff, 0xfeff8000,
  0xffffffff, 0xfeff7fff, 0xffffffff, 0xfeff0000, 0xffffffff, 0xfefeffff,
  0xffffffff, 0xfe000000, 0xffffffff, 0xfdffffff, 0xffffffff, 0xa8cb1d24,
  0xffffffff, 0x47892a0a, 0xfffffffe, 0xff000000, 0xfffffffe, 0xfeffffff,
  0xfffeffff, 0xff000000, 0xfffeffff, 0xfeffffff, 0xffefffff, 0xff000000,
  0xffefffff, 0xfeffffff, 0xa9cb1d24, 0x47892a0a, 0x7fffffff, 0xff000000,
  0x7fffffff, 0xff000000, 0x48892a0a, 0xa8cb1d25, 0x000fffff, 0xff000001,
  0x000fffff, 0xff000000, 0x0000ffff, 0xff000001, 0x0000ffff, 0xff000000,
  0x00000000, 0xff000001, 0x00000000, 0xff000000, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0xffffffff, 0xff010001, 0xffffffff, 0xff010000,
  0xffffffff, 0xff008001, 0xffffffff, 0xff008000, 0xffffffff, 0xff000002,
  0xffffffff, 0xff000001, 0xffffffff, 0xff000000, 0xffffffff, 0xfeffffff,
  0xffffffff, 0xfefffffe, 0xffffffff, 0xfeff8001, 0xffffffff, 0xfeff8000,
  0xffffffff, 0xfeff0001, 0xffffffff, 0xfeff0000, 0xffffffff, 0xfe000001,
  0xffffffff, 0xfe000000, 0xffffffff, 0xa8cb1d25, 0xffffffff, 0x47892a0b,
  0xfffffffe, 0xff000001, 0xfffffffe, 0xff000000, 0xfffeffff, 0xff000001,
  0xfffeffff, 0xff000000, 0xffefffff, 0xff000001, 0xffefffff, 0xff000000,
  0xa9cb1d24, 0x47892a0b, 0x7fffffff, 0xff000001, 0x7fffffff, 0xfffeffff,
  0x48892a0a, 0xa9ca1d24, 0x000fffff, 0xffff0000, 0x000fffff, 0xfffeffff,
  0x0000ffff, 0xffff0000, 0x0000ffff, 0xfffeffff, 0x00000000, 0xffff0000,
  0x00000000, 0xfffeffff, 0x00000000, 0x00ff0000, 0x00000000, 0x00feffff,
  0x00000000, 0x00000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffff8000,
  0xffffffff, 0xffff7fff, 0xffffffff, 0xffff0001, 0xffffffff, 0xffff0000,
  0xffffffff, 0xfffeffff, 0xffffffff, 0xfffefffe, 0xffffffff, 0xfffefffd,
  0xffffffff, 0xfffe8000, 0xffffffff, 0xfffe7fff, 0xffffffff, 0xfffe0000,
  0xffffffff, 0xfffdffff, 0xffffffff, 0xfeff0000, 0xffffffff, 0xfefeffff,
  0xffffffff, 0xa9ca1d24, 0xffffffff, 0x48882a0a, 0xfffffffe, 0xffff0000,
  0xfffffffe, 0xfffeffff, 0xfffeffff, 0xffff0000, 0xfffeffff, 0xfffeffff,
  0xffefffff, 0xffff0000, 0xffefffff, 0xfffeffff, 0xa9cb1d24, 0x48882a0a,
  0x7fffffff, 0xffff0000, 0x7fffffff, 0xffff0000, 0x48892a0a, 0xa9ca1d25,
  0x000fffff, 0xffff0001, 0x000fffff, 0xffff0000, 0x0000ffff, 0xffff0001,
  0x0000ffff, 0xffff0000, 0x00000000, 0xffff0001, 0x00000000, 0xffff0000,
  0x00000000, 0x00ff0001, 0x00000000, 0x00ff0000, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0xffffffff, 0xffff8001, 0xffffffff, 0xffff8000,
  0xffffffff, 0xffff0002, 0xffffffff, 0xffff0001, 0xffffffff, 0xffff0000,
  0xffffffff, 0xfffeffff, 0xffffffff, 0xfffefffe, 0xffffffff, 0xfffe8001,
  0xffffffff, 0xfffe8000, 0xffffffff, 0xfffe0001, 0xffffffff, 0xfffe0000,
  0xffffffff, 0xfeff0001, 0xffffffff, 0xfeff0000, 0xffffffff, 0xa9ca1d25,
  0xffffffff, 0x48882a0b, 0xfffffffe, 0xffff0001, 0xfffffffe, 0xffff0000,
  0xfffeffff, 0xffff0001, 0xfffeffff, 0xffff0000, 0xffefffff, 0xffff0001,
  0xffefffff, 0xffff0000, 0xa9cb1d24, 0x48882a0b, 0x7fffffff, 0xffff0001,
  0x7fffffff, 0xffff7fff, 0x48892a0a, 0xa9ca9d24, 0x000fffff, 0xffff8000,
  0x000fffff, 0xffff7fff, 0x0000ffff, 0xffff8000, 0x0000ffff, 0xffff7fff,
  0x00000000, 0xffff8000, 0x00000000, 0xffff7fff, 0x00000000, 0x00ff8000,
  0x00000000, 0x00ff7fff, 0x00000000, 0x00008000, 0x00000000, 0x00007fff,
  0x00000000, 0x00000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffff8001,
  0xffffffff, 0xffff8000, 0xffffffff, 0xffff7fff, 0xffffffff, 0xffff7ffe,
  0xffffffff, 0xffff7ffd, 0xffffffff, 0xffff0000, 0xffffffff, 0xfffeffff,
  0xffffffff, 0xfffe8000, 0xffffffff, 0xfffe7fff, 0xffffffff, 0xfeff8000,
  0xffffffff, 0xfeff7fff, 0xffffffff, 0xa9ca9d24, 0xffffffff, 0x4888aa0a,
  0xfffffffe, 0xffff8000, 0xfffffffe, 0xffff7fff, 0xfffeffff, 0xffff8000,
  0xfffeffff, 0xffff7fff, 0xffefffff, 0xffff8000, 0xffefffff, 0xffff7fff,
  0xa9cb1d24, 0x4888aa0a, 0x7fffffff, 0xffff8000, 0x7fffffff, 0xffff8000,
  0x48892a0a, 0xa9ca9d25, 0x000fffff, 0xffff8001, 0x000fffff, 0xffff8000,
  0x0000ffff, 0xffff8001, 0x0000ffff, 0xffff8000, 0x00000000, 0xffff8001,
  0x00000000, 0xffff8000, 0x00000000, 0x00ff8001, 0x00000000, 0x00ff8000,
  0x00000000, 0x00008001, 0x00000000, 0x00008000, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0xffffffff, 0xffff8002, 0xffffffff, 0xffff8001,
  0xffffffff, 0xffff8000, 0xffffffff, 0xffff7fff, 0xffffffff, 0xffff7ffe,
  0xffffffff, 0xffff0001, 0xffffffff, 0xffff0000, 0xffffffff, 0xfffe8001,
  0xffffffff, 0xfffe8000, 0xffffffff, 0xfeff8001, 0xffffffff, 0xfeff8000,
  0xffffffff, 0xa9ca9d25, 0xffffffff, 0x4888aa0b, 0xfffffffe, 0xffff8001,
  0xfffffffe, 0xffff8000, 0xfffeffff, 0xffff8001, 0xfffeffff, 0xffff8000,
  0xffefffff, 0xffff8001, 0xffefffff, 0xffff8000, 0xa9cb1d24, 0x4888aa0b,
  0x7fffffff, 0xffff8001, 0x7fffffff, 0xfffffffe, 0x48892a0a, 0xa9cb1d23,
  0x000fffff, 0xffffffff, 0x000fffff, 0xfffffffe, 0x0000ffff, 0xffffffff,
  0x0000ffff, 0xfffffffe, 0x00000000, 0xffffffff, 0x00000000, 0xfffffffe,
  0x00000000, 0x00ffffff, 0x00000000, 0x00fffffe, 0x00000000, 0x0000ffff,
  0x00000000, 0x0000fffe, 0x00000000, 0x00007fff, 0x00000000, 0x00007ffe,
  0x00000000, 0x00000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xfffffffe,
  0xffffffff, 0xfffffffd, 0xffffffff, 0xfffffffc, 0xffffffff, 0xffff7fff,
  0xffffffff, 0xffff7ffe, 0xffffffff, 0xfffeffff, 0xffffffff, 0xfffefffe,
  0xffffffff, 0xfeffffff, 0xffffffff, 0xfefffffe, 0xffffffff, 0xa9cb1d23,
  0xffffffff, 0x48892a09, 0xfffffffe, 0xffffffff, 0xfffffffe, 0xfffffffe,
  0xfffeffff, 0xffffffff, 0xfffeffff, 0xfffffffe, 0xffefffff, 0xffffffff,
  0xffefffff, 0xfffffffe, 0xa9cb1d24, 0x48892a09, 0x7fffffff, 0xffffffff,
  0x7fffffff, 0xffffffff, 0x48892a0a, 0xa9cb1d24, 0x00100000, 0x00000000,
  0x000fffff, 0xffffffff, 0x00010000, 0x00000000, 0x0000ffff, 0xffffffff,
  0x00000001, 0x00000000, 0x00000000, 0xffffffff, 0x00000000, 0x01000000,
  0x00000000, 0x00ffffff, 0x00000000, 0x00010000, 0x00000000, 0x0000ffff,
  0x00000000, 0x00008000, 0x00000000, 0x00007fff, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xfffffffe,
  0xffffffff, 0xfffffffd, 0xffffffff, 0xffff8000, 0xffffffff, 0xffff7fff,
  0xffffffff, 0xffff0000, 0xffffffff, 0xfffeffff, 0xffffffff, 0xff000000,
  0xffffffff, 0xfeffffff, 0xffffffff, 0xa9cb1d24, 0xffffffff, 0x48892a0a,
  0xffffffff, 0x00000000, 0xfffffffe, 0xffffffff, 0xffff0000, 0x00000000,
  0xfffeffff, 0xffffffff, 0xfff00000, 0x00000000, 0xffefffff, 0xffffffff,
  0xa9cb1d24, 0x48892a0a, 0x80000000, 0x00000000, 0x80000000, 0x00000000,
  0x48892a0a, 0xa9cb1d25, 0x00100000, 0x00000001, 0x00100000, 0x00000000,
  0x00010000, 0x00000001, 0x00010000, 0x00000000, 0x00000001, 0x00000001,
  0x00000001, 0x00000000, 0x00000000, 0x01000001, 0x00000000, 0x01000000,
  0x00000000, 0x00010001, 0x00000000, 0x00010000, 0x00000000, 0x00008001,
  0x00000000, 0x00008000, 0x00000000, 0x00000002, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xfffffffe,
  0xffffffff, 0xffff8001, 0xffffffff, 0xffff8000, 0xffffffff, 0xffff0001,
  0xffffffff, 0xffff0000, 0xffffffff, 0xff000001, 0xffffffff, 0xff000000,
  0xffffffff, 0xa9cb1d25, 0xffffffff, 0x48892a0b, 0xffffffff, 0x00000001,
  0xffffffff, 0x00000000, 0xffff0000, 0x00000001, 0xffff0000, 0x00000000,
  0xfff00000, 0x00000001, 0xfff00000, 0x00000000, 0xa9cb1d24, 0x48892a0b,
  0x80000000, 0x00000001, 0x80000000, 0x00000001, 0x48892a0a, 0xa9cb1d26,
  0x00100000, 0x00000002, 0x00100000, 0x00000001, 0x00010000, 0x00000002,
  0x00010000, 0x00000001, 0x00000001, 0x00000002, 0x00000001, 0x00000001,
  0x00000000, 0x01000002, 0x00000000, 0x01000001, 0x00000000, 0x00010002,
  0x00000000, 0x00010001, 0x00000000, 0x00008002, 0x00000000, 0x00008001,
  0x00000000, 0x00000003, 0x00000000, 0x00000002, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffff8002,
  0xffffffff, 0xffff8001, 0xffffffff, 0xffff0002, 0xffffffff, 0xffff0001,
  0xffffffff, 0xff000002, 0xffffffff, 0xff000001, 0xffffffff, 0xa9cb1d26,
  0xffffffff, 0x48892a0c, 0xffffffff, 0x00000002, 0xffffffff, 0x00000001,
  0xffff0000, 0x00000002, 0xffff0000, 0x00000001, 0xfff00000, 0x00000002,
  0xfff00000, 0x00000001, 0xa9cb1d24, 0x48892a0c, 0x80000000, 0x00000002,
  0x80000000, 0x00000002, 0x48892a0a, 0xa9cb1d27, 0x00100000, 0x00000003,
  0x00100000, 0x00000002, 0x00010000, 0x00000003, 0x00010000, 0x00000002,
  0x00000001, 0x00000003, 0x00000001, 0x00000002, 0x00000000, 0x01000003,
  0x00000000, 0x01000002, 0x00000000, 0x00010003, 0x00000000, 0x00010002,
  0x00000000, 0x00008003, 0x00000000, 0x00008002, 0x00000000, 0x00000004,
  0x00000000, 0x00000003, 0x00000000, 0x00000002, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0xffffffff, 0xffff8003, 0xffffffff, 0xffff8002,
  0xffffffff, 0xffff0003, 0xffffffff, 0xffff0002, 0xffffffff, 0xff000003,
  0xffffffff, 0xff000002, 0xffffffff, 0xa9cb1d27, 0xffffffff, 0x48892a0d,
  0xffffffff, 0x00000003, 0xffffffff, 0x00000002, 0xffff0000, 0x00000003,
  0xffff0000, 0x00000002, 0xfff00000, 0x00000003, 0xfff00000, 0x00000002,
  0xa9cb1d24, 0x48892a0d, 0x80000000, 0x00000003, 0x80000000, 0x00007fff,
  0x48892a0a, 0xa9cb9d24, 0x00100000, 0x00008000, 0x00100000, 0x00007fff,
  0x00010000, 0x00008000, 0x00010000, 0x00007fff, 0x00000001, 0x00008000,
  0x00000001, 0x00007fff, 0x00000000, 0x01008000, 0x00000000, 0x01007fff,
  0x00000000, 0x00018000, 0x00000000, 0x00017fff, 0x00000000, 0x00010000,
  0x00000000, 0x0000ffff, 0x00000000, 0x00008001, 0x00000000, 0x00008000,
  0x00000000, 0x00007fff, 0x00000000, 0x00007ffe, 0x00000000, 0x00007ffd,
  0x00000000, 0x00000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffff8000,
  0xffffffff, 0xffff7fff, 0xffffffff, 0xff008000, 0xffffffff, 0xff007fff,
  0xffffffff, 0xa9cb9d24, 0xffffffff, 0x4889aa0a, 0xffffffff, 0x00008000,
  0xffffffff, 0x00007fff, 0xffff0000, 0x00008000, 0xffff0000, 0x00007fff,
  0xfff00000, 0x00008000, 0xfff00000, 0x00007fff, 0xa9cb1d24, 0x4889aa0a,
  0x80000000, 0x00008000, 0x80000000, 0x00008000, 0x48892a0a, 0xa9cb9d25,
  0x00100000, 0x00008001, 0x00100000, 0x00008000, 0x00010000, 0x00008001,
  0x00010000, 0x00008000, 0x00000001, 0x00008001, 0x00000001, 0x00008000,
  0x00000000, 0x01008001, 0x00000000, 0x01008000, 0x00000000, 0x00018001,
  0x00000000, 0x00018000, 0x00000000, 0x00010001, 0x00000000, 0x00010000,
  0x00000000, 0x00008002, 0x00000000, 0x00008001, 0x00000000, 0x00008000,
  0x00000000, 0x00007fff, 0x00000000, 0x00007ffe, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0xffffffff, 0xffff8001, 0xffffffff, 0xffff8000,
  0xffffffff, 0xff008001, 0xffffffff, 0xff008000, 0xffffffff, 0xa9cb9d25,
  0xffffffff, 0x4889aa0b, 0xffffffff, 0x00008001, 0xffffffff, 0x00008000,
  0xffff0000, 0x00008001, 0xffff0000, 0x00008000, 0xfff00000, 0x00008001,
  0xfff00000, 0x00008000, 0xa9cb1d24, 0x4889aa0b, 0x80000000, 0x00008001,
  0x80000000, 0x0000ffff, 0x48892a0a, 0xa9cc1d24, 0x00100000, 0x00010000,
  0x00100000, 0x0000ffff, 0x00010000, 0x00010000, 0x00010000, 0x0000ffff,
  0x00000001, 0x00010000, 0x00000001, 0x0000ffff, 0x00000000, 0x01010000,
  0x00000000, 0x0100ffff, 0x00000000, 0x00020000, 0x00000000, 0x0001ffff,
  0x00000000, 0x00018000, 0x00000000, 0x00017fff, 0x00000000, 0x00010001,
  0x00000000, 0x00010000, 0x00000000, 0x0000ffff, 0x00000000, 0x0000fffe,
  0x00000000, 0x0000fffd, 0x00000000, 0x00008000, 0x00000000, 0x00007fff,
  0x00000000, 0x00000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xff010000,
  0xffffffff, 0xff00ffff, 0xffffffff, 0xa9cc1d24, 0xffffffff, 0x488a2a0a,
  0xffffffff, 0x00010000, 0xffffffff, 0x0000ffff, 0xffff0000, 0x00010000,
  0xffff0000, 0x0000ffff, 0xfff00000, 0x00010000, 0xfff00000, 0x0000ffff,
  0xa9cb1d24, 0x488a2a0a, 0x80000000, 0x00010000, 0x80000000, 0x00010000,
  0x48892a0a, 0xa9cc1d25, 0x00100000, 0x00010001, 0x00100000, 0x00010000,
  0x00010000, 0x00010001, 0x00010000, 0x00010000, 0x00000001, 0x00010001,
  0x00000001, 0x00010000, 0x00000000, 0x01010001, 0x00000000, 0x01010000,
  0x00000000, 0x00020001, 0x00000000, 0x00020000, 0x00000000, 0x00018001,
  0x00000000, 0x00018000, 0x00000000, 0x00010002, 0x00000000, 0x00010001,
  0x00000000, 0x00010000, 0x00000000, 0x0000ffff, 0x00000000, 0x0000fffe,
  0x00000000, 0x00008001, 0x00000000, 0x00008000, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0xffffffff, 0xff010001, 0xffffffff, 0xff010000,
  0xffffffff, 0xa9cc1d25, 0xffffffff, 0x488a2a0b, 0xffffffff, 0x00010001,
  0xffffffff, 0x00010000, 0xffff0000, 0x00010001, 0xffff0000, 0x00010000,
  0xfff00000, 0x00010001, 0xfff00000, 0x00010000, 0xa9cb1d24, 0x488a2a0b,
  0x80000000, 0x00010001, 0x80000000, 0x00ffffff, 0x48892a0a, 0xaacb1d24,
  0x00100000, 0x01000000, 0x00100000, 0x00ffffff, 0x00010000, 0x01000000,
  0x00010000, 0x00ffffff, 0x00000001, 0x01000000, 0x00000001, 0x00ffffff,
  0x00000000, 0x02000000, 0x00000000, 0x01ffffff, 0x00000000, 0x01010000,
  0x00000000, 0x0100ffff, 0x00000000, 0x01008000, 0x00000000, 0x01007fff,
  0x00000000, 0x01000001, 0x00000000, 0x01000000, 0x00000000, 0x00ffffff,
  0x00000000, 0x00fffffe, 0x00000000, 0x00fffffd, 0x00000000, 0x00ff8000,
  0x00000000, 0x00ff7fff, 0x00000000, 0x00ff0000, 0x00000000, 0x00feffff,
  0x00000000, 0x00000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xaacb1d24,
  0xffffffff, 0x49892a0a, 0xffffffff, 0x01000000, 0xffffffff, 0x00ffffff,
  0xffff0000, 0x01000000, 0xffff0000, 0x00ffffff, 0xfff00000, 0x01000000,
  0xfff00000, 0x00ffffff, 0xa9cb1d24, 0x49892a0a, 0x80000000, 0x01000000,
  0x80000000, 0x01000000, 0x48892a0a, 0xaacb1d25, 0x00100000, 0x01000001,
  0x00100000, 0x01000000, 0x00010000, 0x01000001, 0x00010000, 0x01000000,
  0x00000001, 0x01000001, 0x00000001, 0x01000000, 0x00000000, 0x02000001,
  0x00000000, 0x02000000, 0x00000000, 0x01010001, 0x00000000, 0x01010000,
  0x00000000, 0x01008001, 0x00000000, 0x01008000, 0x00000000, 0x01000002,
  0x00000000, 0x01000001, 0x00000000, 0x01000000, 0x00000000, 0x00ffffff,
  0x00000000, 0x00fffffe, 0x00000000, 0x00ff8001, 0x00000000, 0x00ff8000,
  0x00000000, 0x00ff0001, 0x00000000, 0x00ff0000, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0xffffffff, 0xaacb1d25, 0xffffffff, 0x49892a0b,
  0xffffffff, 0x01000001, 0xffffffff, 0x01000000, 0xffff0000, 0x01000001,
  0xffff0000, 0x01000000, 0xfff00000, 0x01000001, 0xfff00000, 0x01000000,
  0xa9cb1d24, 0x49892a0b, 0x80000000, 0x01000001, 0x80000000, 0x5634e2db,
  0x48892a0b, 0x00000000, 0x00100000, 0x5634e2dc, 0x00100000, 0x5634e2db,
  0x00010000, 0x5634e2dc, 0x00010000, 0x5634e2db, 0x00000001, 0x5634e2dc,
  0x00000001, 0x5634e2db, 0x00000000, 0x5734e2dc, 0x00000000, 0x5734e2db,
  0x00000000, 0x5635e2dc, 0x00000000, 0x5635e2db, 0x00000000, 0x563562dc,
  0x00000000, 0x563562db, 0x00000000, 0x5634e2dd, 0x00000000, 0x5634e2dc,
  0x00000000, 0x5634e2db, 0x00000000, 0x5634e2da, 0x00000000, 0x5634e2d9,
  0x00000000, 0x563462dc, 0x00000000, 0x563462db, 0x00000000, 0x5633e2dc,
  0x00000000, 0x5633e2db, 0x00000000, 0x5534e2dc, 0x00000000, 0x5534e2db,
  0x00000000, 0x00000000, 0xffffffff, 0x9ebe0ce6, 0xffffffff, 0x5634e2dc,
  0xffffffff, 0x5634e2db, 0xffff0000, 0x5634e2dc, 0xffff0000, 0x5634e2db,
  0xfff00000, 0x5634e2dc, 0xfff00000, 0x5634e2db, 0xa9cb1d24, 0x9ebe0ce6,
  0x80000000, 0x5634e2dc, 0x80000000, 0xb776d5f5, 0x48892a0b, 0x6141f31a,
  0x00100000, 0xb776d5f6, 0x00100000, 0xb776d5f5, 0x00010000, 0xb776d5f6,
  0x00010000, 0xb776d5f5, 0x00000001, 0xb776d5f6, 0x00000001, 0xb776d5f5,
  0x00000000, 0xb876d5f6, 0x00000000, 0xb876d5f5, 0x00000000, 0xb777d5f6,
  0x00000000, 0xb777d5f5, 0x00000000, 0xb77755f6, 0x00000000, 0xb77755f5,
  0x00000000, 0xb776d5f7, 0x00000000, 0xb776d5f6, 0x00000000, 0xb776d5f5,
  0x00000000, 0xb776d5f4, 0x00000000, 0xb776d5f3, 0x00000000, 0xb77655f6,
  0x00000000, 0xb77655f5, 0x00000000, 0xb775d5f6, 0x00000000, 0xb775d5f5,
  0x00000000, 0xb676d5f6, 0x00000000, 0xb676d5f5, 0x00000000, 0x6141f31a,
  0x00000000, 0x00000000, 0xffffffff, 0xb776d5f6, 0xffffffff, 0xb776d5f5,
  0xffff0000, 0xb776d5f6, 0xffff0000, 0xb776d5f5, 0xfff00000, 0xb776d5f6,
  0xfff00000, 0xb776d5f5, 0xa9cb1d25, 0x00000000, 0x80000000, 0xb776d5f6,
  0x80000000, 0xffffffff, 0x48892a0b, 0xa9cb1d24, 0x00100001, 0x00000000,
  0x00100000, 0xffffffff, 0x00010001, 0x00000000, 0x00010000, 0xffffffff,
  0x00000002, 0x00000000, 0x00000001, 0xffffffff, 0x00000001, 0x01000000,
  0x00000001, 0x00ffffff, 0x00000001, 0x00010000, 0x00000001, 0x0000ffff,
  0x00000001, 0x00008000, 0x00000001, 0x00007fff, 0x00000001, 0x00000001,
  0x00000001, 0x00000000, 0x00000000, 0xffffffff, 0x00000000, 0xfffffffe,
  0x00000000, 0xfffffffd, 0x00000000, 0xffff8000, 0x00000000, 0xffff7fff,
  0x00000000, 0xffff0000, 0x00000000, 0xfffeffff, 0x00000000, 0xff000000,
  0x00000000, 0xfeffffff, 0x00000000, 0xa9cb1d24, 0x00000000, 0x48892a0a,
  0x00000000, 0x00000000, 0xffffffff, 0xffffffff, 0xffff0001, 0x00000000,
  0xffff0000, 0xffffffff, 0xfff00001, 0x00000000, 0xfff00000, 0xffffffff,
  0xa9cb1d25, 0x48892a0a, 0x80000001, 0x00000000, 0x80000001, 0x00000000,
  0x48892a0b, 0xa9cb1d25, 0x00100001, 0x00000001, 0x00100001, 0x00000000,
  0x00010001, 0x00000001, 0x00010001, 0x00000000, 0x00000002, 0x00000001,
  0x00000002, 0x00000000, 0x00000001, 0x01000001, 0x00000001, 0x01000000,
  0x00000001, 0x00010001, 0x00000001, 0x00010000, 0x00000001, 0x00008001,
  0x00000001, 0x00008000, 0x00000001, 0x00000002, 0x00000001, 0x00000001,
  0x00000001, 0x00000000, 0x00000000, 0xffffffff, 0x00000000, 0xfffffffe,
  0x00000000, 0xffff8001, 0x00000000, 0xffff8000, 0x00000000, 0xffff0001,
  0x00000000, 0xffff0000, 0x00000000, 0xff000001, 0x00000000, 0xff000000,
  0x00000000, 0xa9cb1d25, 0x00000000, 0x48892a0b, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0xffff0001, 0x00000001, 0xffff0001, 0x00000000,
  0xfff00001, 0x00000001, 0xfff00001, 0x00000000, 0xa9cb1d25, 0x48892a0b,
  0x80000001, 0x00000001, 0x8000ffff, 0xffffffff, 0x488a2a0a, 0xa9cb1d24,
  0x00110000, 0x00000000, 0x0010ffff, 0xffffffff, 0x00020000, 0x00000000,
  0x0001ffff, 0xffffffff, 0x00010001, 0x00000000, 0x00010000, 0xffffffff,
  0x00010000, 0x01000000, 0x00010000, 0x00ffffff, 0x00010000, 0x00010000,
  0x00010000, 0x0000ffff, 0x00010000, 0x00008000, 0x00010000, 0x00007fff,
  0x00010000, 0x00000001, 0x00010000, 0x00000000, 0x0000ffff, 0xffffffff,
  0x0000ffff, 0xfffffffe, 0x0000ffff, 0xfffffffd, 0x0000ffff, 0xffff8000,
  0x0000ffff, 0xffff7fff, 0x0000ffff, 0xffff0000, 0x0000ffff, 0xfffeffff,
  0x0000ffff, 0xff000000, 0x0000ffff, 0xfeffffff, 0x0000ffff, 0xa9cb1d24,
  0x0000ffff, 0x48892a0a, 0x0000ffff, 0x00000000, 0x0000fffe, 0xffffffff,
  0x00000000, 0x00000000, 0xffffffff, 0xffffffff, 0xfff10000, 0x00000000,
  0xfff0ffff, 0xffffffff, 0xa9cc1d24, 0x48892a0a, 0x80010000, 0x00000000,
  0x80010000, 0x00000000, 0x488a2a0a, 0xa9cb1d25, 0x00110000, 0x00000001,
  0x00110000, 0x00000000, 0x00020000, 0x00000001, 0x00020000, 0x00000000,
  0x00010001, 0x00000001, 0x00010001, 0x00000000, 0x00010000, 0x01000001,
  0x00010000, 0x01000000, 0x00010000, 0x00010001, 0x00010000, 0x00010000,
  0x00010000, 0x00008001, 0x00010000, 0x00008000, 0x00010000, 0x00000002,
  0x00010000, 0x00000001, 0x00010000, 0x00000000, 0x0000ffff, 0xffffffff,
  0x0000ffff, 0xfffffffe, 0x0000ffff, 0xffff8001, 0x0000ffff, 0xffff8000,
  0x0000ffff, 0xffff0001, 0x0000ffff, 0xffff0000, 0x0000ffff, 0xff000001,
  0x0000ffff, 0xff000000, 0x0000ffff, 0xa9cb1d25, 0x0000ffff, 0x48892a0b,
  0x0000ffff, 0x00000001, 0x0000ffff, 0x00000000, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0xfff10000, 0x00000001, 0xfff10000, 0x00000000,
  0xa9cc1d24, 0x48892a0b, 0x80010000, 0x00000001, 0x800fffff, 0xffffffff,
  0x48992a0a, 0xa9cb1d24, 0x00200000, 0x00000000, 0x001fffff, 0xffffffff,
  0x00110000, 0x00000000, 0x0010ffff, 0xffffffff, 0x00100001, 0x00000000,
  0x00100000, 0xffffffff, 0x00100000, 0x01000000, 0x00100000, 0x00ffffff,
  0x00100000, 0x00010000, 0x00100000, 0x0000ffff, 0x00100000, 0x00008000,
  0x00100000, 0x00007fff, 0x00100000, 0x00000001, 0x00100000, 0x00000000,
  0x000fffff, 0xffffffff, 0x000fffff, 0xfffffffe, 0x000fffff, 0xfffffffd,
  0x000fffff, 0xffff8000, 0x000fffff, 0xffff7fff, 0x000fffff, 0xffff0000,
  0x000fffff, 0xfffeffff, 0x000fffff, 0xff000000, 0x000fffff, 0xfeffffff,
  0x000fffff, 0xa9cb1d24, 0x000fffff, 0x48892a0a, 0x000fffff, 0x00000000,
  0x000ffffe, 0xffffffff, 0x000f0000, 0x00000000, 0x000effff, 0xffffffff,
  0x00000000, 0x00000000, 0xffffffff, 0xffffffff, 0xa9db1d24, 0x48892a0a,
  0x80100000, 0x00000000, 0x80100000, 0x00000000, 0x48992a0a, 0xa9cb1d25,
  0x00200000, 0x00000001, 0x00200000, 0x00000000, 0x00110000, 0x00000001,
  0x00110000, 0x00000000, 0x00100001, 0x00000001, 0x00100001, 0x00000000,
  0x00100000, 0x01000001, 0x00100000, 0x01000000, 0x00100000, 0x00010001,
  0x00100000, 0x00010000, 0x00100000, 0x00008001, 0x00100000, 0x00008000,
  0x00100000, 0x00000002, 0x00100000, 0x00000001, 0x00100000, 0x00000000,
  0x000fffff, 0xffffffff, 0x000fffff, 0xfffffffe, 0x000fffff, 0xffff8001,
  0x000fffff, 0xffff8000, 0x000fffff, 0xffff0001, 0x000fffff, 0xffff0000,
  0x000fffff, 0xff000001, 0x000fffff, 0xff000000, 0x000fffff, 0xa9cb1d25,
  0x000fffff, 0x48892a0b, 0x000fffff, 0x00000001, 0x000fffff, 0x00000000,
  0x000f0000, 0x00000001, 0x000f0000, 0x00000000, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0xa9db1d24, 0x48892a0b, 0x80100000, 0x00000001,
  0xd634e2db, 0xb776d5f5, 0x9ebe0ce6, 0x6141f31a, 0x5644e2db, 0xb776d5f6,
  0x5644e2db, 0xb776d5f5, 0x5635e2db, 0xb776d5f6, 0x5635e2db, 0xb776d5f5,
  0x5634e2dc, 0xb776d5f6, 0x5634e2dc, 0xb776d5f5, 0x5634e2db, 0xb876d5f6,
  0x5634e2db, 0xb876d5f5, 0x5634e2db, 0xb777d5f6, 0x5634e2db, 0xb777d5f5,
  0x5634e2db, 0xb77755f6, 0x5634e2db, 0xb77755f5, 0x5634e2db, 0xb776d5f7,
  0x5634e2db, 0xb776d5f6, 0x5634e2db, 0xb776d5f5, 0x5634e2db, 0xb776d5f4,
  0x5634e2db, 0xb776d5f3, 0x5634e2db, 0xb77655f6, 0x5634e2db, 0xb77655f5,
  0x5634e2db, 0xb775d5f6, 0x5634e2db, 0xb775d5f5, 0x5634e2db, 0xb676d5f6,
  0x5634e2db, 0xb676d5f5, 0x5634e2db, 0x6141f31a, 0x5634e2db, 0x00000000,
  0x5634e2da, 0xb776d5f6, 0x5634e2da, 0xb776d5f5, 0x5633e2db, 0xb776d5f6,
  0x5633e2db, 0xb776d5f5, 0x5624e2db, 0xb776d5f6, 0x5624e2db, 0xb776d5f5,
  0x00000000, 0x00000000, 0xd634e2db, 0xb776d5f6, 0xffffffff, 0xffffffff,
  0xc8892a0a, 0xa9cb1d24, 0x80100000, 0x00000000, 0x800fffff, 0xffffffff,
  0x80010000, 0x00000000, 0x8000ffff, 0xffffffff, 0x80000001, 0x00000000,
  0x80000000, 0xffffffff, 0x80000000, 0x01000000, 0x80000000, 0x00ffffff,
  0x80000000, 0x00010000, 0x80000000, 0x0000ffff, 0x80000000, 0x00008000,
  0x80000000, 0x00007fff, 0x80000000, 0x00000001, 0x80000000, 0x00000000,
  0x7fffffff, 0xffffffff, 0x7fffffff, 0xfffffffe, 0x7fffffff, 0xfffffffd,
  0x7fffffff, 0xffff8000, 0x7fffffff, 0xffff7fff, 0x7fffffff, 0xffff0000,
  0x7fffffff, 0xfffeffff, 0x7fffffff, 0xff000000, 0x7fffffff, 0xfeffffff,
  0x7fffffff, 0xa9cb1d24, 0x7fffffff, 0x48892a0a, 0x7fffffff, 0x00000000,
  0x7ffffffe, 0xffffffff, 0x7fff0000, 0x00000000, 0x7ffeffff, 0xffffffff,
  0x7ff00000, 0x00000000, 0x7fefffff, 0xffffffff, 0x29cb1d24, 0x48892a0a,
  0x00000000, 0x00000000
];
toInt32s(TEST_SUB_BITS);

var TEST_MUL_BITS = [
  0x80000000, 0x00000000, 0x80000000, 0x00000000, 0x1ad92a0a, 0xa9cb1d25,
  0x00000000, 0x00000000, 0xd2500000, 0x00000000, 0x00100000, 0x00000000,
  0x80000000, 0x00000000, 0x65ae2a0a, 0xa9cb1d25, 0x00110000, 0x00000001,
  0x00100000, 0x00000000, 0x00000000, 0x00000000, 0x1d250000, 0x00000000,
  0x00010000, 0x00000000, 0x00000000, 0x00000000, 0x00010000, 0x00000000,
  0x80000000, 0x00000000, 0xf254472f, 0xa9cb1d25, 0x00100001, 0x00000001,
  0x00100000, 0x00000000, 0x00010001, 0x00000001, 0x00010000, 0x00000000,
  0x00000000, 0x00000000, 0xa9cb1d25, 0x00000000, 0x00000001, 0x00000000,
  0x00000000, 0x00000000, 0x00000001, 0x00000000, 0x00000000, 0x00000000,
  0x00000001, 0x00000000, 0x80000000, 0x00000000, 0x5332f527, 0xcecb1d25,
  0x00100000, 0x01000001, 0x00100000, 0x00000000, 0x00010000, 0x01000001,
  0x00010000, 0x00000000, 0x01000001, 0x01000001, 0x01000001, 0x00000000,
  0x00000000, 0x00000000, 0x0aa9cb1d, 0x25000000, 0x00000000, 0x01000000,
  0x00000000, 0x00000000, 0x00000000, 0x01000000, 0x00000000, 0x00000000,
  0x01000000, 0x01000000, 0x01000000, 0x00000000, 0x00010000, 0x01000000,
  0x80000000, 0x00000000, 0x7293d3d5, 0xc6f01d25, 0x00100000, 0x00010001,
  0x00100000, 0x00000000, 0x00010000, 0x00010001, 0x00010000, 0x00000000,
  0x00010001, 0x00010001, 0x00010001, 0x00000000, 0x00000100, 0x01010001,
  0x00000100, 0x01000000, 0x00000000, 0x00000000, 0x2a0aa9cb, 0x1d250000,
  0x00000000, 0x00010000, 0x00000000, 0x00000000, 0x00000000, 0x00010000,
  0x00000000, 0x00000000, 0x00010000, 0x00010000, 0x00010000, 0x00000000,
  0x00000100, 0x00010000, 0x00000100, 0x00000000, 0x00000001, 0x00010000,
  0x80000000, 0x00000000, 0xdd8e7ef0, 0x385d9d25, 0x00100000, 0x00008001,
  0x00100000, 0x00000000, 0x80010000, 0x00008001, 0x80010000, 0x00000000,
  0x00008001, 0x00008001, 0x00008001, 0x00000000, 0x00000080, 0x01008001,
  0x00000080, 0x01000000, 0x00000000, 0x80018001, 0x00000000, 0x80010000,
  0x00000000, 0x00000000, 0x950554e5, 0x8e928000, 0x00000000, 0x00008000,
  0x00000000, 0x00000000, 0x80000000, 0x00008000, 0x80000000, 0x00000000,
  0x00008000, 0x00008000, 0x00008000, 0x00000000, 0x00000080, 0x00008000,
  0x00000080, 0x00000000, 0x00000000, 0x80008000, 0x00000000, 0x80000000,
  0x00000000, 0x40008000, 0x00000000, 0x00000000, 0x91125415, 0x53963a4a,
  0x00200000, 0x00000002, 0x00200000, 0x00000000, 0x00020000, 0x00000002,
  0x00020000, 0x00000000, 0x00000002, 0x00000002, 0x00000002, 0x00000000,
  0x00000000, 0x02000002, 0x00000000, 0x02000000, 0x00000000, 0x00020002,
  0x00000000, 0x00020000, 0x00000000, 0x00010002, 0x00000000, 0x00010000,
  0x80000000, 0x00000000, 0x48892a0a, 0xa9cb1d25, 0x00100000, 0x00000001,
  0x00100000, 0x00000000, 0x00010000, 0x00000001, 0x00010000, 0x00000000,
  0x00000001, 0x00000001, 0x00000001, 0x00000000, 0x00000000, 0x01000001,
  0x00000000, 0x01000000, 0x00000000, 0x00010001, 0x00000000, 0x00010000,
  0x00000000, 0x00008001, 0x00000000, 0x00008000, 0x00000000, 0x00000002,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x80000000, 0x00000000, 0xb776d5f5, 0x5634e2db,
  0xffefffff, 0xffffffff, 0xfff00000, 0x00000000, 0xfffeffff, 0xffffffff,
  0xffff0000, 0x00000000, 0xfffffffe, 0xffffffff, 0xffffffff, 0x00000000,
  0xffffffff, 0xfeffffff, 0xffffffff, 0xff000000, 0xffffffff, 0xfffeffff,
  0xffffffff, 0xffff0000, 0xffffffff, 0xffff7fff, 0xffffffff, 0xffff8000,
  0xffffffff, 0xfffffffe, 0xffffffff, 0xffffffff, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x6eedabea, 0xac69c5b6, 0xffdfffff, 0xfffffffe,
  0xffe00000, 0x00000000, 0xfffdffff, 0xfffffffe, 0xfffe0000, 0x00000000,
  0xfffffffd, 0xfffffffe, 0xfffffffe, 0x00000000, 0xffffffff, 0xfdfffffe,
  0xffffffff, 0xfe000000, 0xffffffff, 0xfffdfffe, 0xffffffff, 0xfffe0000,
  0xffffffff, 0xfffefffe, 0xffffffff, 0xffff0000, 0xffffffff, 0xfffffffc,
  0xffffffff, 0xfffffffe, 0x00000000, 0x00000000, 0x00000000, 0x00000002,
  0x80000000, 0x00000000, 0xb383d525, 0x1b389d25, 0x000fffff, 0xffff8001,
  0x00100000, 0x00000000, 0x8000ffff, 0xffff8001, 0x80010000, 0x00000000,
  0xffff8000, 0xffff8001, 0xffff8001, 0x00000000, 0xffffff80, 0x00ff8001,
  0xffffff80, 0x01000000, 0xffffffff, 0x80008001, 0xffffffff, 0x80010000,
  0xffffffff, 0xc0000001, 0xffffffff, 0xc0008000, 0xffffffff, 0xffff0002,
  0xffffffff, 0xffff8001, 0x00000000, 0x00000000, 0x00000000, 0x00007fff,
  0x00000000, 0x0000fffe, 0x00000000, 0x00000000, 0x6afaab1a, 0x716d8000,
  0xffffffff, 0xffff8000, 0x00000000, 0x00000000, 0x7fffffff, 0xffff8000,
  0x80000000, 0x00000000, 0xffff7fff, 0xffff8000, 0xffff8000, 0x00000000,
  0xffffff7f, 0xffff8000, 0xffffff80, 0x00000000, 0xffffffff, 0x7fff8000,
  0xffffffff, 0x80000000, 0xffffffff, 0xbfff8000, 0xffffffff, 0xc0000000,
  0xffffffff, 0xffff0000, 0xffffffff, 0xffff8000, 0x00000000, 0x00000000,
  0x00000000, 0x00008000, 0x00000000, 0x00010000, 0x00000000, 0x3fff8000,
  0x80000000, 0x00000000, 0x1e7e803f, 0x8ca61d25, 0x000fffff, 0xffff0001,
  0x00100000, 0x00000000, 0x0000ffff, 0xffff0001, 0x00010000, 0x00000000,
  0xffff0000, 0xffff0001, 0xffff0001, 0x00000000, 0xffffff00, 0x00ff0001,
  0xffffff00, 0x01000000, 0xffffffff, 0x00000001, 0xffffffff, 0x00010000,
  0xffffffff, 0x7fff8001, 0xffffffff, 0x80008000, 0xffffffff, 0xfffe0002,
  0xffffffff, 0xffff0001, 0x00000000, 0x00000000, 0x00000000, 0x0000ffff,
  0x00000000, 0x0001fffe, 0x00000000, 0x7ffe8001, 0x00000000, 0x7fff8000,
  0x00000000, 0x00000000, 0xd5f55634, 0xe2db0000, 0xffffffff, 0xffff0000,
  0x00000000, 0x00000000, 0xffffffff, 0xffff0000, 0x00000000, 0x00000000,
  0xfffeffff, 0xffff0000, 0xffff0000, 0x00000000, 0xfffffeff, 0xffff0000,
  0xffffff00, 0x00000000, 0xfffffffe, 0xffff0000, 0xffffffff, 0x00000000,
  0xffffffff, 0x7fff0000, 0xffffffff, 0x80000000, 0xffffffff, 0xfffe0000,
  0xffffffff, 0xffff0000, 0x00000000, 0x00000000, 0x00000000, 0x00010000,
  0x00000000, 0x00020000, 0x00000000, 0x7fff0000, 0x00000000, 0x80000000,
  0x00000000, 0xffff0000, 0x80000000, 0x00000000, 0x3ddf5eed, 0x84cb1d25,
  0x000fffff, 0xff000001, 0x00100000, 0x00000000, 0x0000ffff, 0xff000001,
  0x00010000, 0x00000000, 0xff000000, 0xff000001, 0xff000001, 0x00000000,
  0xffff0000, 0x00000001, 0xffff0000, 0x01000000, 0xfffffeff, 0xff010001,
  0xffffff00, 0x00010000, 0xffffff7f, 0xff008001, 0xffffff80, 0x00008000,
  0xffffffff, 0xfe000002, 0xffffffff, 0xff000001, 0x00000000, 0x00000000,
  0x00000000, 0x00ffffff, 0x00000000, 0x01fffffe, 0x0000007f, 0xfeff8001,
  0x0000007f, 0xffff8000, 0x000000ff, 0xfeff0001, 0x000000ff, 0xffff0000,
  0x00000000, 0x00000000, 0xf55634e2, 0xdb000000, 0xffffffff, 0xff000000,
  0x00000000, 0x00000000, 0xffffffff, 0xff000000, 0x00000000, 0x00000000,
  0xfeffffff, 0xff000000, 0xff000000, 0x00000000, 0xfffeffff, 0xff000000,
  0xffff0000, 0x00000000, 0xfffffeff, 0xff000000, 0xffffff00, 0x00000000,
  0xffffff7f, 0xff000000, 0xffffff80, 0x00000000, 0xffffffff, 0xfe000000,
  0xffffffff, 0xff000000, 0x00000000, 0x00000000, 0x00000000, 0x01000000,
  0x00000000, 0x02000000, 0x0000007f, 0xff000000, 0x00000080, 0x00000000,
  0x000000ff, 0xff000000, 0x00000100, 0x00000000, 0x0000ffff, 0xff000000,
  0x80000000, 0x00000000, 0xbc56e5ef, 0x15ff6759, 0xd24fffff, 0xa9cb1d25,
  0xd2500000, 0x00000000, 0x1d24ffff, 0xa9cb1d25, 0x1d250000, 0x00000000,
  0xa9cb1d24, 0xa9cb1d25, 0xa9cb1d25, 0x00000000, 0xffa9cb1c, 0xcecb1d25,
  0xffa9cb1d, 0x25000000, 0xffffa9ca, 0xc6f01d25, 0xffffa9cb, 0x1d250000,
  0xffffd4e5, 0x385d9d25, 0xffffd4e5, 0x8e928000, 0xffffffff, 0x53963a4a,
  0xffffffff, 0xa9cb1d25, 0x00000000, 0x00000000, 0x00000000, 0x5634e2db,
  0x00000000, 0xac69c5b6, 0x00002b1a, 0x1b389d25, 0x00002b1a, 0x716d8000,
  0x00005634, 0x8ca61d25, 0x00005634, 0xe2db0000, 0x005634e2, 0x84cb1d25,
  0x005634e2, 0xdb000000, 0x80000000, 0x00000000, 0x74756f10, 0x9f4f5297,
  0xa0afffff, 0x48892a0b, 0xa0b00000, 0x00000000, 0x2a0affff, 0x48892a0b,
  0x2a0b0000, 0x00000000, 0x48892a0a, 0x48892a0b, 0x48892a0b, 0x00000000,
  0xff488929, 0x53892a0b, 0xff48892a, 0x0b000000, 0xffff4888, 0x72942a0b,
  0xffff4889, 0x2a0b0000, 0xffffa443, 0xdd8eaa0b, 0xffffa444, 0x95058000,
  0xfffffffe, 0x91125416, 0xffffffff, 0x48892a0b, 0x00000000, 0x00000000,
  0x00000000, 0xb776d5f5, 0x00000001, 0x6eedabea, 0x00005bba, 0xb383aa0b,
  0x00005bbb, 0x6afa8000, 0x0000b776, 0x1e7e2a0b, 0x0000b776, 0xd5f50000,
  0x00b776d5, 0x3d892a0b, 0x00b776d5, 0xf5000000, 0x3dc7d297, 0x9f4f5297,
  0x80000000, 0x00000000, 0x9ebe0ce5, 0xa9cb1d25, 0x000fffff, 0x00000001,
  0x00100000, 0x00000000, 0x0000ffff, 0x00000001, 0x00010000, 0x00000000,
  0x00000000, 0x00000001, 0x00000001, 0x00000000, 0xfeffffff, 0x01000001,
  0xff000000, 0x01000000, 0xfffeffff, 0x00010001, 0xffff0000, 0x00010000,
  0xffff7fff, 0x00008001, 0xffff8000, 0x00008000, 0xfffffffe, 0x00000002,
  0xffffffff, 0x00000001, 0x00000000, 0x00000000, 0x00000000, 0xffffffff,
  0x00000001, 0xfffffffe, 0x00007ffe, 0xffff8001, 0x00007fff, 0xffff8000,
  0x0000fffe, 0xffff0001, 0x0000ffff, 0xffff0000, 0x00fffffe, 0xff000001,
  0x00ffffff, 0xff000000, 0x5634e2da, 0xa9cb1d25, 0xb776d5f4, 0x48892a0b,
  0x00000000, 0x00000000, 0x5634e2db, 0x00000000, 0xffffffff, 0x00000000,
  0x00000000, 0x00000000, 0xffffffff, 0x00000000, 0x00000000, 0x00000000,
  0xffffffff, 0x00000000, 0x00000000, 0x00000000, 0xfeffffff, 0x00000000,
  0xff000000, 0x00000000, 0xfffeffff, 0x00000000, 0xffff0000, 0x00000000,
  0xffff7fff, 0x00000000, 0xffff8000, 0x00000000, 0xfffffffe, 0x00000000,
  0xffffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000001, 0x00000000,
  0x00000002, 0x00000000, 0x00007fff, 0x00000000, 0x00008000, 0x00000000,
  0x0000ffff, 0x00000000, 0x00010000, 0x00000000, 0x00ffffff, 0x00000000,
  0x01000000, 0x00000000, 0x5634e2db, 0x00000000, 0xb776d5f5, 0x00000000,
  0xffffffff, 0x00000000, 0x80000000, 0x00000000, 0x2b642a0a, 0xa9cb1d25,
  0x000f0000, 0x00000001, 0x00100000, 0x00000000, 0x00000000, 0x00000001,
  0x00010000, 0x00000000, 0xffff0001, 0x00000001, 0x00000001, 0x00000000,
  0xffff0000, 0x01000001, 0x00000000, 0x01000000, 0xffff0000, 0x00010001,
  0x00000000, 0x00010000, 0x7fff0000, 0x00008001, 0x80000000, 0x00008000,
  0xfffe0000, 0x00000002, 0xffff0000, 0x00000001, 0x00000000, 0x00000000,
  0x0000ffff, 0xffffffff, 0x0001ffff, 0xfffffffe, 0x7ffeffff, 0xffff8001,
  0x7fffffff, 0xffff8000, 0xfffeffff, 0xffff0001, 0xffffffff, 0xffff0000,
  0xfffeffff, 0xff000001, 0xffffffff, 0xff000000, 0xe2daffff, 0xa9cb1d25,
  0xd5f4ffff, 0x48892a0b, 0xfffeffff, 0x00000001, 0xffffffff, 0x00000000,
  0x00000000, 0x00000000, 0xe2db0000, 0x00000000, 0xffff0000, 0x00000000,
  0x00000000, 0x00000000, 0xffff0000, 0x00000000, 0x00000000, 0x00000000,
  0xffff0000, 0x00000000, 0x00000000, 0x00000000, 0xffff0000, 0x00000000,
  0x00000000, 0x00000000, 0xffff0000, 0x00000000, 0x00000000, 0x00000000,
  0x7fff0000, 0x00000000, 0x80000000, 0x00000000, 0xfffe0000, 0x00000000,
  0xffff0000, 0x00000000, 0x00000000, 0x00000000, 0x00010000, 0x00000000,
  0x00020000, 0x00000000, 0x7fff0000, 0x00000000, 0x80000000, 0x00000000,
  0xffff0000, 0x00000000, 0x00000000, 0x00000000, 0xffff0000, 0x00000000,
  0x00000000, 0x00000000, 0xe2db0000, 0x00000000, 0xd5f50000, 0x00000000,
  0xffff0000, 0x00000000, 0x00000000, 0x00000000, 0xffff0000, 0x00000000,
  0x80000000, 0x00000000, 0x76392a0a, 0xa9cb1d25, 0x00000000, 0x00000001,
  0x00100000, 0x00000000, 0xfff10000, 0x00000001, 0x00010000, 0x00000000,
  0xfff00001, 0x00000001, 0x00000001, 0x00000000, 0xfff00000, 0x01000001,
  0x00000000, 0x01000000, 0xfff00000, 0x00010001, 0x00000000, 0x00010000,
  0xfff00000, 0x00008001, 0x00000000, 0x00008000, 0xffe00000, 0x00000002,
  0xfff00000, 0x00000001, 0x00000000, 0x00000000, 0x000fffff, 0xffffffff,
  0x001fffff, 0xfffffffe, 0xffefffff, 0xffff8001, 0xffffffff, 0xffff8000,
  0xffefffff, 0xffff0001, 0xffffffff, 0xffff0000, 0xffefffff, 0xff000001,
  0xffffffff, 0xff000000, 0x2dafffff, 0xa9cb1d25, 0x5f4fffff, 0x48892a0b,
  0xffefffff, 0x00000001, 0xffffffff, 0x00000000, 0xffef0000, 0x00000001,
  0xffff0000, 0x00000000, 0x00000000, 0x00000000, 0x2db00000, 0x00000000,
  0xfff00000, 0x00000000, 0x00000000, 0x00000000, 0xfff00000, 0x00000000,
  0x00000000, 0x00000000, 0xfff00000, 0x00000000, 0x00000000, 0x00000000,
  0xfff00000, 0x00000000, 0x00000000, 0x00000000, 0xfff00000, 0x00000000,
  0x00000000, 0x00000000, 0xfff00000, 0x00000000, 0x00000000, 0x00000000,
  0xffe00000, 0x00000000, 0xfff00000, 0x00000000, 0x00000000, 0x00000000,
  0x00100000, 0x00000000, 0x00200000, 0x00000000, 0xfff00000, 0x00000000,
  0x00000000, 0x00000000, 0xfff00000, 0x00000000, 0x00000000, 0x00000000,
  0xfff00000, 0x00000000, 0x00000000, 0x00000000, 0x2db00000, 0x00000000,
  0x5f500000, 0x00000000, 0xfff00000, 0x00000000, 0x00000000, 0x00000000,
  0xfff00000, 0x00000000, 0x00000000, 0x00000000, 0xfff00000, 0x00000000,
  0x80000000, 0x00000000, 0x8a74d669, 0x9f4f5297, 0x4a7b1d24, 0x48892a0b,
  0xa0b00000, 0x00000000, 0xd3d61d24, 0x48892a0b, 0x2a0b0000, 0x00000000,
  0xf254472f, 0x48892a0b, 0x48892a0b, 0x00000000, 0xce13a64e, 0x53892a0b,
  0x2448892a, 0x0b000000, 0xc6ef65ad, 0x72942a0b, 0x1d244889, 0x2a0b0000,
  0x385d4168, 0xdd8eaa0b, 0x8e922444, 0x95058000, 0x53963a48, 0x91125416,
  0xa9cb1d24, 0x48892a0b, 0x00000000, 0x00000000, 0x5634e2db, 0xb776d5f5,
  0xac69c5b7, 0x6eedabea, 0x1b38f8df, 0xb383aa0b, 0x716ddbbb, 0x6afa8000,
  0x8ca6d49b, 0x1e7e2a0b, 0xe2dbb776, 0xd5f50000, 0x858293fa, 0x3d892a0b,
  0xdbb776d5, 0xf5000000, 0x53c739f0, 0x9f4f5297, 0x22ca6fa5, 0x36ad9c79,
  0x6141f319, 0x48892a0b, 0xb776d5f5, 0x00000000, 0x7fc01d24, 0x48892a0b,
  0xd5f50000, 0x00000000, 0x091b1d24, 0x48892a0b, 0x5f500000, 0x00000000,
  0x80000000, 0x00000000, 0xc8892a0a, 0xa9cb1d25, 0x80100000, 0x00000001,
  0x00100000, 0x00000000, 0x80010000, 0x00000001, 0x00010000, 0x00000000,
  0x80000001, 0x00000001, 0x00000001, 0x00000000, 0x80000000, 0x01000001,
  0x00000000, 0x01000000, 0x80000000, 0x00010001, 0x00000000, 0x00010000,
  0x80000000, 0x00008001, 0x00000000, 0x00008000, 0x00000000, 0x00000002,
  0x80000000, 0x00000001, 0x00000000, 0x00000000, 0x7fffffff, 0xffffffff,
  0xffffffff, 0xfffffffe, 0x7fffffff, 0xffff8001, 0xffffffff, 0xffff8000,
  0x7fffffff, 0xffff0001, 0xffffffff, 0xffff0000, 0x7fffffff, 0xff000001,
  0xffffffff, 0xff000000, 0x7fffffff, 0xa9cb1d25, 0x7fffffff, 0x48892a0b,
  0x7fffffff, 0x00000001, 0xffffffff, 0x00000000, 0x7fff0000, 0x00000001,
  0xffff0000, 0x00000000, 0x7ff00000, 0x00000001, 0xfff00000, 0x00000000,
  0x29cb1d24, 0x48892a0b
];
toInt32s(TEST_MUL_BITS);

var TEST_DIV_BITS = [
  0x00000000, 0x00000001, 0x00000000, 0x00000001, 0x00000000, 0x000007ff,
  0x00000000, 0x00000800, 0x00000000, 0x00007fff, 0x00000000, 0x00008000,
  0x00000000, 0x7fffffff, 0x00000000, 0x80000000, 0x0000007f, 0xffff8000,
  0x00000080, 0x00000000, 0x00007fff, 0x80007fff, 0x00008000, 0x00000000,
  0x0000fffe, 0x0003fff8, 0x00010000, 0x00000000, 0x40000000, 0x00000000,
  0x80000000, 0x00000000, 0x80000000, 0x00000000, 0xc0000000, 0x00000000,
  0xfffefffd, 0xfffbfff8, 0xffff0000, 0x00000000, 0xffff7fff, 0x7fff8000,
  0xffff8000, 0x00000000, 0xffffff7f, 0xffff8000, 0xffffff80, 0x00000000,
  0xfffffffe, 0x83e3cc1a, 0xffffffff, 0x4d64985a, 0xffffffff, 0x80000000,
  0xffffffff, 0x80000000, 0xffffffff, 0xffff8000, 0xffffffff, 0xffff8000,
  0xffffffff, 0xfffff800, 0xffffffff, 0xfffff800, 0xffffffff, 0xffffffff,
  0xffffffff, 0xffffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000001,
  0x00000000, 0x00000488, 0x00000000, 0x00000488, 0x00000000, 0x00004889,
  0x00000000, 0x00004889, 0x00000000, 0x48892a0a, 0x00000000, 0x48892a0a,
  0x00000048, 0x8929c220, 0x00000048, 0x892a0aa9, 0x00004888, 0xe181c849,
  0x00004889, 0x2a0aa9cb, 0x00009111, 0x31f2efb0, 0x00009112, 0x54155396,
  0x24449505, 0x54e58e92, 0x48892a0a, 0xa9cb1d25, 0xb776d5f5, 0x5634e2db,
  0xdbbb6afa, 0xab1a716e, 0xffff6eec, 0x89c3bff2, 0xffff6eed, 0xabeaac6a,
  0xffffb776, 0x8d6be3a1, 0xffffb776, 0xd5f55635, 0xffffffb7, 0x76d5acce,
  0xffffffb7, 0x76d5f557, 0xffffffff, 0x2898cfc6, 0xffffffff, 0x9ac930b4,
  0xffffffff, 0xb776d5f6, 0xffffffff, 0xb776d5f6, 0xffffffff, 0xffffb777,
  0xffffffff, 0xffffb777, 0xffffffff, 0xfffffb78, 0xffffffff, 0xfffffb78,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000001, 0x00000000, 0x00000001,
  0x00000000, 0x0000000f, 0x00000000, 0x00000010, 0x00000000, 0x000fffff,
  0x00000000, 0x00100000, 0x00000000, 0x0ffffff0, 0x00000000, 0x10000000,
  0x0000000f, 0xfff0000f, 0x00000010, 0x00000000, 0x0000001f, 0xffc0007f,
  0x00000020, 0x00000000, 0x00080000, 0x00000000, 0x00100000, 0x00000001,
  0xffefffff, 0xffffffff, 0xfff80000, 0x00000000, 0xffffffdf, 0xffbfff80,
  0xffffffe0, 0x00000000, 0xffffffef, 0xffeffff0, 0xfffffff0, 0x00000000,
  0xffffffff, 0xeffffff0, 0xffffffff, 0xf0000000, 0xffffffff, 0xffd07c7a,
  0xffffffff, 0xffe9ac94, 0xffffffff, 0xfff00000, 0xffffffff, 0xfff00000,
  0xffffffff, 0xfffffff0, 0xffffffff, 0xfffffff0, 0xffffffff, 0xffffffff,
  0xffffffff, 0xffffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000001, 0x00000000, 0x0000000f, 0x00000000, 0x00000010,
  0x00000000, 0x000fffff, 0x00000000, 0x00100000, 0x00000000, 0x0ffffff0,
  0x00000000, 0x10000000, 0x0000000f, 0xfff0000f, 0x00000010, 0x00000000,
  0x0000001f, 0xffc0007f, 0x00000020, 0x00000000, 0x00080000, 0x00000000,
  0x00100000, 0x00000000, 0xfff00000, 0x00000000, 0xfff80000, 0x00000000,
  0xffffffdf, 0xffbfff80, 0xffffffe0, 0x00000000, 0xffffffef, 0xffeffff0,
  0xfffffff0, 0x00000000, 0xffffffff, 0xeffffff0, 0xffffffff, 0xf0000000,
  0xffffffff, 0xffd07c7a, 0xffffffff, 0xffe9ac94, 0xffffffff, 0xfff00000,
  0xffffffff, 0xfff00000, 0xffffffff, 0xfffffff0, 0xffffffff, 0xfffffff0,
  0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000001,
  0x00000000, 0x00000001, 0x00000000, 0x0000ffff, 0x00000000, 0x00010000,
  0x00000000, 0x00ffffff, 0x00000000, 0x01000000, 0x00000000, 0xffff0001,
  0x00000001, 0x00000000, 0x00000001, 0xfffc0007, 0x00000002, 0x00000000,
  0x00008000, 0x00000000, 0x00010000, 0x00000001, 0xfffeffff, 0xffffffff,
  0xffff8000, 0x00000000, 0xfffffffd, 0xfffbfff8, 0xfffffffe, 0x00000000,
  0xfffffffe, 0xfffeffff, 0xffffffff, 0x00000000, 0xffffffff, 0xfeffffff,
  0xffffffff, 0xff000000, 0xffffffff, 0xfffd07c8, 0xffffffff, 0xfffe9aca,
  0xffffffff, 0xffff0000, 0xffffffff, 0xffff0000, 0xffffffff, 0xffffffff,
  0xffffffff, 0xffffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000001, 0x00000000, 0x0000ffff,
  0x00000000, 0x00010000, 0x00000000, 0x00ffffff, 0x00000000, 0x01000000,
  0x00000000, 0xffff0000, 0x00000001, 0x00000000, 0x00000001, 0xfffc0007,
  0x00000002, 0x00000000, 0x00008000, 0x00000000, 0x00010000, 0x00000000,
  0xffff0000, 0x00000000, 0xffff8000, 0x00000000, 0xfffffffd, 0xfffbfff8,
  0xfffffffe, 0x00000000, 0xfffffffe, 0xfffeffff, 0xffffffff, 0x00000000,
  0xffffffff, 0xfeffffff, 0xffffffff, 0xff000000, 0xffffffff, 0xfffd07c8,
  0xffffffff, 0xfffe9aca, 0xffffffff, 0xffff0000, 0xffffffff, 0xffff0000,
  0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000001, 0x00000000, 0x00000001, 0x00000000, 0x000000ff,
  0x00000000, 0x00000100, 0x00000000, 0x0000ffff, 0x00000000, 0x00010000,
  0x00000000, 0x0001fffc, 0x00000000, 0x00020000, 0x00000000, 0x80000000,
  0x00000001, 0x00000001, 0xfffffffe, 0xffffffff, 0xffffffff, 0x80000000,
  0xffffffff, 0xfffdfffc, 0xffffffff, 0xfffe0000, 0xffffffff, 0xfffeffff,
  0xffffffff, 0xffff0000, 0xffffffff, 0xffffff00, 0xffffffff, 0xffffff00,
  0xffffffff, 0xfffffffe, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
  0xffffffff, 0xffffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000001,
  0x00000000, 0x000000ff, 0x00000000, 0x00000100, 0x00000000, 0x0000ffff,
  0x00000000, 0x00010000, 0x00000000, 0x0001fffc, 0x00000000, 0x00020000,
  0x00000000, 0x80000000, 0x00000001, 0x00000000, 0xffffffff, 0x00000000,
  0xffffffff, 0x80000000, 0xffffffff, 0xfffdfffc, 0xffffffff, 0xfffe0000,
  0xffffffff, 0xfffeffff, 0xffffffff, 0xffff0000, 0xffffffff, 0xffffff00,
  0xffffffff, 0xffffff00, 0xffffffff, 0xfffffffe, 0xffffffff, 0xffffffff,
  0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000001, 0x00000000, 0x00000001,
  0x00000000, 0x000000ff, 0x00000000, 0x00000100, 0x00000000, 0x000001ff,
  0x00000000, 0x00000200, 0x00000000, 0x00800000, 0x00000000, 0x01000001,
  0xffffffff, 0xfeffffff, 0xffffffff, 0xff800000, 0xffffffff, 0xfffffe00,
  0xffffffff, 0xfffffe00, 0xffffffff, 0xffffff00, 0xffffffff, 0xffffff00,
  0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000001, 0x00000000, 0x000000ff, 0x00000000, 0x00000100,
  0x00000000, 0x000001ff, 0x00000000, 0x00000200, 0x00000000, 0x00800000,
  0x00000000, 0x01000000, 0xffffffff, 0xff000000, 0xffffffff, 0xff800000,
  0xffffffff, 0xfffffe00, 0xffffffff, 0xfffffe00, 0xffffffff, 0xffffff00,
  0xffffffff, 0xffffff00, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000001,
  0x00000000, 0x00000001, 0x00000000, 0x00000001, 0x00000000, 0x00000002,
  0x00000000, 0x00008000, 0x00000000, 0x00010001, 0xffffffff, 0xfffeffff,
  0xffffffff, 0xffff8000, 0xffffffff, 0xfffffffe, 0xffffffff, 0xfffffffe,
  0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000001, 0x00000000, 0x00000001,
  0x00000000, 0x00000002, 0x00000000, 0x00008000, 0x00000000, 0x00010000,
  0xffffffff, 0xffff0000, 0xffffffff, 0xffff8000, 0xffffffff, 0xfffffffe,
  0xffffffff, 0xfffffffe, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000001, 0x00000000, 0x00000001, 0x00000000, 0x00004000,
  0x00000000, 0x00008001, 0xffffffff, 0xffff7fff, 0xffffffff, 0xffffc000,
  0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000001,
  0x00000000, 0x00004000, 0x00000000, 0x00008000, 0xffffffff, 0xffff8000,
  0xffffffff, 0xffffc000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000001, 0x00000000, 0x00000002,
  0xffffffff, 0xfffffffe, 0xffffffff, 0xffffffff, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000001, 0xffffffff, 0xffffffff, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xffffffff,
  0x00000000, 0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xffffffff,
  0xffffffff, 0xfffffffe, 0x00000000, 0x00000002, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0xffffffff, 0xffffc001, 0xffffffff, 0xffff8001, 0x00000000, 0x00007fff,
  0x00000000, 0x00003fff, 0x00000000, 0x00000001, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0xffffffff, 0xffffffff, 0xffffffff, 0xffffc000, 0xffffffff, 0xffff8000,
  0x00000000, 0x00008000, 0x00000000, 0x00004000, 0x00000000, 0x00000001,
  0x00000000, 0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffff8001,
  0xffffffff, 0xffff0001, 0x00000000, 0x0000ffff, 0x00000000, 0x00007fff,
  0x00000000, 0x00000002, 0x00000000, 0x00000001, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xfffffffe,
  0xffffffff, 0xffff8000, 0xffffffff, 0xffff0000, 0x00000000, 0x00010000,
  0x00000000, 0x00008000, 0x00000000, 0x00000002, 0x00000000, 0x00000002,
  0x00000000, 0x00000001, 0x00000000, 0x00000001, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0xffffffff, 0xffffff01, 0xffffffff, 0xffffff01, 0xffffffff, 0xfffffe01,
  0xffffffff, 0xfffffe01, 0xffffffff, 0xff800001, 0xffffffff, 0xff000001,
  0x00000000, 0x00ffffff, 0x00000000, 0x007fffff, 0x00000000, 0x00000200,
  0x00000000, 0x000001ff, 0x00000000, 0x00000100, 0x00000000, 0x000000ff,
  0x00000000, 0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0xffffffff, 0xffffffff, 0xffffffff, 0xffffff01, 0xffffffff, 0xffffff00,
  0xffffffff, 0xfffffe01, 0xffffffff, 0xfffffe00, 0xffffffff, 0xff800000,
  0xffffffff, 0xff000000, 0x00000000, 0x01000000, 0x00000000, 0x00800000,
  0x00000000, 0x00000200, 0x00000000, 0x00000200, 0x00000000, 0x00000100,
  0x00000000, 0x00000100, 0x00000000, 0x00000001, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0xffffffff, 0xffffffaa, 0xffffffff, 0xffffffaa, 0xffffffff, 0xffffa9cc,
  0xffffffff, 0xffffa9cc, 0xffffffff, 0xffff5398, 0xffffffff, 0xffff5397,
  0xffffffff, 0xd4e58e93, 0xffffffff, 0xa9cb1d25, 0x00000000, 0x5634e2db,
  0x00000000, 0x2b1a716d, 0x00000000, 0x0000ac6b, 0x00000000, 0x0000ac69,
  0x00000000, 0x00005635, 0x00000000, 0x00005634, 0x00000000, 0x00000056,
  0x00000000, 0x00000056, 0x00000000, 0x00000001, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0xffffffff, 0xffffff49, 0xffffffff, 0xffffff49,
  0xffffffff, 0xffff488a, 0xffffffff, 0xffff488a, 0xffffffff, 0xfffe9116,
  0xffffffff, 0xfffe9113, 0xffffffff, 0xa4449506, 0xffffffff, 0x48892a0b,
  0x00000000, 0xb776d5f5, 0x00000000, 0x5bbb6afa, 0x00000000, 0x00016ef0,
  0x00000000, 0x00016eed, 0x00000000, 0x0000b777, 0x00000000, 0x0000b776,
  0x00000000, 0x000000b7, 0x00000000, 0x000000b7, 0x00000000, 0x00000002,
  0x00000000, 0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xffffff01,
  0xffffffff, 0xffffff01, 0xffffffff, 0xffff0001, 0xffffffff, 0xffff0001,
  0xffffffff, 0xfffe0004, 0xffffffff, 0xfffe0001, 0xffffffff, 0x80000001,
  0xffffffff, 0x00000001, 0x00000000, 0xffffffff, 0x00000000, 0x7fffffff,
  0x00000000, 0x00020004, 0x00000000, 0x0001ffff, 0x00000000, 0x00010001,
  0x00000000, 0x0000ffff, 0x00000000, 0x00000100, 0x00000000, 0x000000ff,
  0x00000000, 0x00000002, 0x00000000, 0x00000001, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xffffffff,
  0xffffffff, 0xffffff01, 0xffffffff, 0xffffff00, 0xffffffff, 0xffff0001,
  0xffffffff, 0xffff0000, 0xffffffff, 0xfffe0004, 0xffffffff, 0xfffe0000,
  0xffffffff, 0x80000000, 0xffffffff, 0x00000000, 0x00000001, 0x00000000,
  0x00000000, 0x80000000, 0x00000000, 0x00020004, 0x00000000, 0x00020000,
  0x00000000, 0x00010001, 0x00000000, 0x00010000, 0x00000000, 0x00000100,
  0x00000000, 0x00000100, 0x00000000, 0x00000002, 0x00000000, 0x00000001,
  0x00000000, 0x00000001, 0x00000000, 0x00000001, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xffff0001,
  0xffffffff, 0xffff0001, 0xffffffff, 0xff000001, 0xffffffff, 0xff000001,
  0xffffffff, 0x00010000, 0xffffffff, 0x00000001, 0xfffffffe, 0x0003fff9,
  0xfffffffe, 0x00000001, 0xffff8000, 0x00000001, 0xffff0000, 0x00000001,
  0x0000ffff, 0xffffffff, 0x00007fff, 0xffffffff, 0x00000002, 0x00040008,
  0x00000001, 0xffffffff, 0x00000001, 0x00010001, 0x00000000, 0xffffffff,
  0x00000000, 0x01000001, 0x00000000, 0x00ffffff, 0x00000000, 0x0002f838,
  0x00000000, 0x00016536, 0x00000000, 0x00010000, 0x00000000, 0x0000ffff,
  0x00000000, 0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xffffffff,
  0xffffffff, 0xffff0001, 0xffffffff, 0xffff0000, 0xffffffff, 0xff000001,
  0xffffffff, 0xff000000, 0xffffffff, 0x00010000, 0xffffffff, 0x00000000,
  0xfffffffe, 0x0003fff9, 0xfffffffe, 0x00000000, 0xffff8000, 0x00000000,
  0xffff0000, 0x00000000, 0x00010000, 0x00000000, 0x00008000, 0x00000000,
  0x00000002, 0x00040008, 0x00000002, 0x00000000, 0x00000001, 0x00010001,
  0x00000001, 0x00000000, 0x00000000, 0x01000001, 0x00000000, 0x01000000,
  0x00000000, 0x0002f838, 0x00000000, 0x00016536, 0x00000000, 0x00010000,
  0x00000000, 0x00010000, 0x00000000, 0x00000001, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xfffffff1,
  0xffffffff, 0xfffffff1, 0xffffffff, 0xfff00001, 0xffffffff, 0xfff00001,
  0xffffffff, 0xf0000010, 0xffffffff, 0xf0000001, 0xfffffff0, 0x000ffff1,
  0xfffffff0, 0x00000001, 0xffffffe0, 0x003fff81, 0xffffffe0, 0x00000001,
  0xfff80000, 0x00000001, 0xfff00000, 0x00000001, 0x000fffff, 0xffffffff,
  0x0007ffff, 0xffffffff, 0x00000020, 0x00400080, 0x0000001f, 0xffffffff,
  0x00000010, 0x00100010, 0x0000000f, 0xffffffff, 0x00000000, 0x10000010,
  0x00000000, 0x0fffffff, 0x00000000, 0x002f8386, 0x00000000, 0x0016536c,
  0x00000000, 0x00100000, 0x00000000, 0x000fffff, 0x00000000, 0x00000010,
  0x00000000, 0x0000000f, 0x00000000, 0x00000001, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xffffffff,
  0xffffffff, 0xfffffff1, 0xffffffff, 0xfffffff0, 0xffffffff, 0xfff00001,
  0xffffffff, 0xfff00000, 0xffffffff, 0xf0000010, 0xffffffff, 0xf0000000,
  0xfffffff0, 0x000ffff1, 0xfffffff0, 0x00000000, 0xffffffe0, 0x003fff81,
  0xffffffe0, 0x00000000, 0xfff80000, 0x00000000, 0xfff00000, 0x00000000,
  0x00100000, 0x00000000, 0x00080000, 0x00000000, 0x00000020, 0x00400080,
  0x00000020, 0x00000000, 0x00000010, 0x00100010, 0x00000010, 0x00000000,
  0x00000000, 0x10000010, 0x00000000, 0x10000000, 0x00000000, 0x002f8386,
  0x00000000, 0x0016536c, 0x00000000, 0x00100000, 0x00000000, 0x00100000,
  0x00000000, 0x00000010, 0x00000000, 0x00000010, 0x00000000, 0x00000001,
  0x00000000, 0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
  0x00000000, 0x00000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xfffffa9d,
  0xffffffff, 0xfffffa9d, 0xffffffff, 0xffffa9cc, 0xffffffff, 0xffffa9cc,
  0xffffffff, 0xa9cb1d25, 0xffffffff, 0xa9cb1d25, 0xffffffa9, 0xcb1d7a7e,
  0xffffffa9, 0xcb1d2449, 0xffffa9cb, 0x7358d531, 0xffffa9cb, 0x1d24488a,
  0xffff5397, 0x93196ae0, 0xffff5396, 0x3a489113, 0xd4e58e92, 0x24449506,
  0xa9cb1d24, 0x48892a0b, 0x5634e2db, 0xb776d5f5, 0x2b1a716d, 0xdbbb6afa,
  0x0000ac6b, 0x1e8dac09, 0x0000ac69, 0xc5b76eed, 0x00005635, 0x3910f087,
  0x00005634, 0xe2dbb776, 0x00000056, 0x34e331ec, 0x00000056, 0x34e2dbb7,
  0x00000001, 0x00000002, 0x00000000, 0x784a3552, 0x00000000, 0x5634e2dc,
  0x00000000, 0x5634e2db, 0x00000000, 0x00005634, 0x00000000, 0x00005634,
  0x00000000, 0x00000563, 0x00000000, 0x00000563, 0x00000000, 0x00000001,
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xffffffff,
  0xffffffff, 0xfffff801, 0xffffffff, 0xfffff801, 0xffffffff, 0xffff8001,
  0xffffffff, 0xffff8001, 0xffffffff, 0x80000001, 0xffffffff, 0x80000001,
  0xffffff80, 0x00008000, 0xffffff80, 0x00000001, 0xffff8000, 0x7fff8001,
  0xffff8000, 0x00000001, 0xffff0001, 0xfffc0008, 0xffff0000, 0x00000001,
  0xc0000000, 0x00000001, 0x80000000, 0x00000001, 0x7fffffff, 0xffffffff,
  0x3fffffff, 0xffffffff, 0x00010002, 0x00040008, 0x0000ffff, 0xffffffff,
  0x00008000, 0x80008000, 0x00007fff, 0xffffffff, 0x00000080, 0x00008000,
  0x0000007f, 0xffffffff, 0x00000001, 0x7c1c33e6, 0x00000000, 0xb29b67a6,
  0x00000000, 0x80000000, 0x00000000, 0x7fffffff, 0x00000000, 0x00008000,
  0x00000000, 0x00007fff, 0x00000000, 0x00000800, 0x00000000, 0x000007ff,
  0x00000000, 0x00000001, 0x00000000, 0x00000001
];
toInt32s(TEST_DIV_BITS);

var TEST_STRINGS = [
  '-9223372036854775808',
  '-5226755067826871589',
  '-4503599627370497',
  '-4503599627370496',
  '-281474976710657',
  '-281474976710656',
  '-4294967297',
  '-4294967296',
  '-16777217',
  '-16777216',
  '-65537',
  '-65536',
  '-32769',
  '-32768',
  '-2',
  '-1',
  '0',
  '1',
  '2',
  '32767',
  '32768',
  '65535',
  '65536',
  '16777215',
  '16777216',
  '1446306523',
  '3078018549',
  '4294967295',
  '4294967296',
  '281474976710655',
  '281474976710656',
  '4503599627370495',
  '4503599627370496',
  '6211839219354490357',
  '9223372036854775807'
];

function setUp() {
  if (Object.seal) {
    Object.seal(goog.math.Long);
  }
}

function testSealingDoesntMakeLazyInitializersUndefined() {
  assertNotNull(goog.math.Long.getZero());
  assertNotNull(goog.math.Long.getOne());
  assertNotNull(goog.math.Long.getNegOne());
  assertNotNull(goog.math.Long.getMaxValue());
  assertNotNull(goog.math.Long.getMinValue());
  assertNotNull(goog.math.Long.getTwoPwr24());
}

function testToFromBits() {
  for (var i = 0; i < TEST_BITS.length; i += 2) {
    var val = goog.math.Long.fromBits(TEST_BITS[i + 1], TEST_BITS[i]);
    assertEquals(TEST_BITS[i], val.getHighBits());
    assertEquals(TEST_BITS[i + 1], val.getLowBits());
  }
}

function testToFromInt() {
  for (var i = 0; i < TEST_BITS.length; i += 1) {
    var val = goog.math.Long.fromInt(TEST_BITS[i]);
    assertEquals(TEST_BITS[i], val.toInt());
  }
}

function testToFromNumber() {
  for (var i = 0; i < TEST_BITS.length; i += 2) {
    var num = TEST_BITS[i] * Math.pow(2, 32) + TEST_BITS[i + 1] >= 0 ?
        TEST_BITS[i + 1] :
        Math.pow(2, 32) + TEST_BITS[i + 1];
    var val = goog.math.Long.fromNumber(num);
    assertEquals(num, val.toNumber());
  }
  // Test edge cases
  assertEquals(goog.math.Long.getZero(), goog.math.Long.fromNumber(NaN));
  assertEquals(
      goog.math.Long.getMaxValue(), goog.math.Long.fromNumber(Infinity));
  assertEquals(
      goog.math.Long.getMinValue(), goog.math.Long.fromNumber(-Infinity));
}

function testIsZero() {
  for (var i = 0; i < TEST_BITS.length; i += 2) {
    var val = goog.math.Long.fromBits(TEST_BITS[i + 1], TEST_BITS[i]);
    assertEquals(TEST_BITS[i] == 0 && TEST_BITS[i + 1] == 0, val.isZero());
  }
}

function testIsNegative() {
  for (var i = 0; i < TEST_BITS.length; i += 2) {
    var val = goog.math.Long.fromBits(TEST_BITS[i + 1], TEST_BITS[i]);
    assertEquals((TEST_BITS[i] >> 31) != 0, val.isNegative());
  }
}

function testIsOdd() {
  for (var i = 0; i < TEST_BITS.length; i += 2) {
    var val = goog.math.Long.fromBits(TEST_BITS[i + 1], TEST_BITS[i]);
    assertEquals((TEST_BITS[i + 1] & 1) != 0, val.isOdd());
  }
}

function createTestComparisons(i) {
  return function() {
    var vi = goog.math.Long.fromBits(TEST_BITS[i + 1], TEST_BITS[i]);
    for (var j = 0; j < TEST_BITS.length; j += 2) {
      var vj = goog.math.Long.fromBits(TEST_BITS[j + 1], TEST_BITS[j]);
      assertEquals(i == j, vi.equals(vj));
      assertEquals(i != j, vi.notEquals(vj));
      assertEquals(i < j, vi.lessThan(vj));
      assertEquals(i <= j, vi.lessThanOrEqual(vj));
      assertEquals(i > j, vi.greaterThan(vj));
      assertEquals(i >= j, vi.greaterThanOrEqual(vj));
    }
  };
}

// Here and below, we translate one conceptual test (e.g., "testComparisons")
// into a number of test functions that will be run separately by jsunit. This
// is necessary because, in some testing configurations, the full combined test
// can take so long that it times out. These smaller tests run much faster.
for (var i = 0; i < TEST_BITS.length; i += 2) {
  goog.global['testComparisons' + i] = createTestComparisons(i);
}

function createTestBitOperations(i) {
  return function() {
    var vi = goog.math.Long.fromBits(TEST_BITS[i + 1], TEST_BITS[i]);
    assertEquals(~TEST_BITS[i], vi.not().getHighBits());
    assertEquals(~TEST_BITS[i + 1], vi.not().getLowBits());

    for (var j = 0; j < TEST_BITS.length; j += 2) {
      var vj = goog.math.Long.fromBits(TEST_BITS[j + 1], TEST_BITS[j]);
      assertEquals(TEST_BITS[i] & TEST_BITS[j], vi.and(vj).getHighBits());
      assertEquals(
          TEST_BITS[i + 1] & TEST_BITS[j + 1], vi.and(vj).getLowBits());
      assertEquals(TEST_BITS[i] | TEST_BITS[j], vi.or(vj).getHighBits());
      assertEquals(TEST_BITS[i + 1] | TEST_BITS[j + 1], vi.or(vj).getLowBits());
      assertEquals(TEST_BITS[i] ^ TEST_BITS[j], vi.xor(vj).getHighBits());
      assertEquals(
          TEST_BITS[i + 1] ^ TEST_BITS[j + 1], vi.xor(vj).getLowBits());
    }

    assertEquals(TEST_BITS[i], vi.shiftLeft(0).getHighBits());
    assertEquals(TEST_BITS[i + 1], vi.shiftLeft(0).getLowBits());
    assertEquals(TEST_BITS[i], vi.shiftRight(0).getHighBits());
    assertEquals(TEST_BITS[i + 1], vi.shiftRight(0).getLowBits());
    assertEquals(TEST_BITS[i], vi.shiftRightUnsigned(0).getHighBits());
    assertEquals(TEST_BITS[i + 1], vi.shiftRightUnsigned(0).getLowBits());

    for (var len = 1; len < 64; ++len) {
      if (len < 32) {
        assertEquals(
            (TEST_BITS[i] << len) | (TEST_BITS[i + 1] >>> (32 - len)),
            vi.shiftLeft(len).getHighBits());
        assertEquals(TEST_BITS[i + 1] << len, vi.shiftLeft(len).getLowBits());

        assertEquals(TEST_BITS[i] >> len, vi.shiftRight(len).getHighBits());
        assertEquals(
            (TEST_BITS[i + 1] >>> len) | (TEST_BITS[i] << (32 - len)),
            vi.shiftRight(len).getLowBits());

        assertEquals(
            TEST_BITS[i] >>> len, vi.shiftRightUnsigned(len).getHighBits());
        assertEquals(
            (TEST_BITS[i + 1] >>> len) | (TEST_BITS[i] << (32 - len)),
            vi.shiftRightUnsigned(len).getLowBits());
      } else {
        assertEquals(
            TEST_BITS[i + 1] << (len - 32), vi.shiftLeft(len).getHighBits());
        assertEquals(0, vi.shiftLeft(len).getLowBits());

        assertEquals(
            TEST_BITS[i] >= 0 ? 0 : -1, vi.shiftRight(len).getHighBits());
        assertEquals(
            TEST_BITS[i] >> (len - 32), vi.shiftRight(len).getLowBits());

        assertEquals(0, vi.shiftRightUnsigned(len).getHighBits());
        if (len == 32) {
          assertEquals(TEST_BITS[i], vi.shiftRightUnsigned(len).getLowBits());
        } else {
          assertEquals(
              TEST_BITS[i] >>> (len - 32),
              vi.shiftRightUnsigned(len).getLowBits());
        }
      }
    }

    assertEquals(TEST_BITS[i], vi.shiftLeft(64).getHighBits());
    assertEquals(TEST_BITS[i + 1], vi.shiftLeft(64).getLowBits());
    assertEquals(TEST_BITS[i], vi.shiftRight(64).getHighBits());
    assertEquals(TEST_BITS[i + 1], vi.shiftRight(64).getLowBits());
    assertEquals(TEST_BITS[i], vi.shiftRightUnsigned(64).getHighBits());
    assertEquals(TEST_BITS[i + 1], vi.shiftRightUnsigned(64).getLowBits());
  };
}

for (var i = 0; i < TEST_BITS.length; i += 2) {
  goog.global['testBitOperations' + i] = createTestBitOperations(i);
}

function testNegation() {
  for (var i = 0; i < TEST_BITS.length; i += 2) {
    var vi = goog.math.Long.fromBits(TEST_BITS[i + 1], TEST_BITS[i]);
    if (TEST_BITS[i + 1] == 0) {
      assertEquals((~TEST_BITS[i] + 1) | 0, vi.negate().getHighBits());
      assertEquals(0, vi.negate().getLowBits());
    } else {
      assertEquals(~TEST_BITS[i], vi.negate().getHighBits());
      assertEquals((~TEST_BITS[i + 1] + 1) | 0, vi.negate().getLowBits());
    }
  }
}

function testAdd() {
  var count = 0;
  for (var i = 0; i < TEST_BITS.length; i += 2) {
    var vi = goog.math.Long.fromBits(TEST_BITS[i + 1], TEST_BITS[i]);
    for (var j = 0; j < i; j += 2) {
      var vj = goog.math.Long.fromBits(TEST_BITS[j + 1], TEST_BITS[j]);
      var result = vi.add(vj);
      assertEquals(TEST_ADD_BITS[count++], result.getHighBits());
      assertEquals(TEST_ADD_BITS[count++], result.getLowBits());
    }
  }
}

function testSubtract() {
  var count = 0;
  for (var i = 0; i < TEST_BITS.length; i += 2) {
    var vi = goog.math.Long.fromBits(TEST_BITS[i + 1], TEST_BITS[i]);
    for (var j = 0; j < TEST_BITS.length; j += 2) {
      var vj = goog.math.Long.fromBits(TEST_BITS[j + 1], TEST_BITS[j]);
      var result = vi.subtract(vj);
      assertEquals(TEST_SUB_BITS[count++], result.getHighBits());
      assertEquals(TEST_SUB_BITS[count++], result.getLowBits());
    }
  }
}

function testMultiply() {
  var count = 0;
  for (var i = 0; i < TEST_BITS.length; i += 2) {
    var vi = goog.math.Long.fromBits(TEST_BITS[i + 1], TEST_BITS[i]);
    for (var j = 0; j < i; j += 2) {
      var vj = goog.math.Long.fromBits(TEST_BITS[j + 1], TEST_BITS[j]);
      var result = vi.multiply(vj);
      assertEquals(TEST_MUL_BITS[count++], result.getHighBits());
      assertEquals(TEST_MUL_BITS[count++], result.getLowBits());
    }
  }
}

function createTestDivMod(i, count) {
  return function() {
    var vi = goog.math.Long.fromBits(TEST_BITS[i + 1], TEST_BITS[i]);
    for (var j = 0; j < TEST_BITS.length; j += 2) {
      var vj = goog.math.Long.fromBits(TEST_BITS[j + 1], TEST_BITS[j]);
      if (!vj.isZero()) {
        var divResult = vi.div(vj);
        assertEquals(TEST_DIV_BITS[count++], divResult.getHighBits());
        assertEquals(TEST_DIV_BITS[count++], divResult.getLowBits());

        var modResult = vi.modulo(vj);
        var combinedResult = divResult.multiply(vj).add(modResult);
        assertTrue(vi.equals(combinedResult));
      }
    }
  }
}

var countPerDivModCall = 0;
for (var j = 0; j < TEST_BITS.length; j += 2) {
  var vj = goog.math.Long.fromBits(TEST_BITS[j + 1], TEST_BITS[j]);
  if (!vj.isZero()) {
    countPerDivModCall += 2;
  }
}

var countDivMod = 0;
for (var i = 0; i < TEST_BITS.length; i += 2) {
  goog.global['testDivMod' + i] = createTestDivMod(i, countDivMod);
  countDivMod += countPerDivModCall;
}

function createTestToFromString(i) {
  return function() {
    var vi = goog.math.Long.fromBits(TEST_BITS[i + 1], TEST_BITS[i]);
    var str = vi.toString(10);
    assertEquals(TEST_STRINGS[i / 2], str);
    assertEquals(
        TEST_BITS[i], goog.math.Long.fromString(str, 10).getHighBits());
    assertEquals(
        TEST_BITS[i + 1], goog.math.Long.fromString(str, 10).getLowBits());

    for (var radix = 2; radix <= 36; ++radix) {
      var result = vi.toString(radix);
      assertEquals(
          TEST_BITS[i], goog.math.Long.fromString(result, radix).getHighBits());
      assertEquals(
          TEST_BITS[i + 1],
          goog.math.Long.fromString(result, radix).getLowBits());
    }
  }
}

for (var i = 0; i < TEST_BITS.length; i += 2) {
  goog.global['testToFromString' + i] = createTestToFromString(i);
}

function testIsStringInRange() {
  var string1 = '9223372036854775808';
  var string2 = '1000000000000000000000000';
  var string3 = '-9223372036854775809';
  var string4 = '-1000000000000000000000000';
  assertEquals(false, goog.math.Long.isStringInRange(string1, 10));
  assertEquals(false, goog.math.Long.isStringInRange(string2, 10));
  assertEquals(false, goog.math.Long.isStringInRange(string3, 10));
  assertEquals(false, goog.math.Long.isStringInRange(string4, 10));

  for (var i = 0; i < TEST_STRINGS.length; i++) {
    assertEquals(true, goog.math.Long.isStringInRange(TEST_STRINGS[i], 10));
  }
}
testIsStringInRange();

// Regression test for
// https://github.com/google/closure-library/pull/498
function testBase36ToString() {
  assertEquals('zzzzzz', goog.math.Long.fromString('zzzzzz', 36).toString(36));
}
